import isEmpty from 'lodash/isEmpty';
import {React, PureComponent, B} from '../base';

import upperFirst from 'lodash/upperFirst';
import {connect} from 'react-redux';

import {CHAR_NBSP} from '../../lib/stringUtils';

import IconGlyph from '../../interfaces/components/IconGlyph';

import {reachGoalOnce} from '../../lib/yaMetrika';
import havePriceAndSeats from '../../lib/segments/tariffs/havePriceAndSeats';
import getTariffKeys from '../../lib/segments/getTariffKeys';
import needToShowTravelTrainTransferButton from './utilities/needToShowTravelTrainTransferButton';

import {setSegmentIsOpened} from '../../actions/search';

import Icon from '../Icon/Icon';
import Spinner from '../Spinner/Spinner';
import Segments from './components/Segments/Segments';
import TransportIcon from '../TransportIcon/TransportIcon';
import TimeAndDuration from './components/TimeAndDuration/TimeAndDuration';
import SegmentBadges from '../SegmentBadges/SegmentBadges';
import ScheduleAndPrices from '../SearchSegment/components/ScheduleAndPrices/ScheduleAndPrices';
import PriceAndActionButton from './components/PriceAndActionButton/PriceAndActionButton';

import segmentTransferKeyset from '../../i18n/segment-transfer';
import checkShowOnlyDirectTrainVariantActions from './utilities/checkShowOnlyDirectTrainVariantActions';

const b = B('SearchTransfer');

const mapStateToProps = ({seoQueryParams, currencies, language, flags}) => ({
    seoQueryParams,
    currencies,
    language,
    flags,
});

class TableSearchTransfer extends PureComponent {
    componentDidMount = () => {
        const {
            data: {segment: transfer},
        } = this.props;
        const {isGone} = transfer;

        reachGoalOnce('transfers_shown');

        if (isGone) {
            return;
        }

        reachGoalOnce('transfersWithPrices_shownTransfer');

        const isInterline = havePriceAndSeats(transfer);
        const transferHavePrices = transfer.segments.some(havePriceAndSeats);
        const transferHaveAllPrices =
            transfer.segments.every(havePriceAndSeats);

        if (transferHavePrices) {
            reachGoalOnce('transfersWithPrices_shownTransferWithPrice');
        }

        if (transferHaveAllPrices) {
            reachGoalOnce('transfersWithPrices_shownTransferWithAllPrices');
        }

        if (isInterline) {
            reachGoalOnce('transfersWithPrices_showInterline');
        }
    };

    onClickToggle = () => {
        const isOpened = !this.isOpened();
        const {
            dispatch,
            data: {
                segment: {id},
            },
        } = this.props;

        if (isOpened) {
            reachGoalOnce('transfer_show_route');
        }

        dispatch(setSegmentIsOpened({id, isOpened}));
    };

    isOpened() {
        const {isOpened} = this.props.data.segment;

        if (typeof isOpened === 'boolean') {
            return isOpened;
        }

        return this.defaultOpened();
    }

    defaultOpened() {
        return this.props.data.numberOfDirectSegments === 0;
    }

    getTransportIcons() {
        const {
            data: {
                segment: {segments},
            },
        } = this.props;

        if (!segments) {
            return null;
        }

        return (
            <React.Fragment>
                {segments.map(({transport, thread}, index) => (
                    <TransportIcon
                        key={index}
                        transportType={transport.code}
                        isExpress={thread && thread.isExpress}
                        isAeroExpress={thread && thread.isAeroExpress}
                    />
                ))}
            </React.Fragment>
        );
    }

    getThread() {
        const {data} = this.props;
        const {segment} = data;

        const {transferTitle} = segment;

        // Если в начале строки стоит предлог, то заменяем первый пробел на неразрывный
        const textForPast =
            transferTitle.slice(0, 4).replace(/\s+/, CHAR_NBSP) +
            transferTitle.slice(4);

        return (
            <div className={b('thread')}>
                <span className={b('title')}>
                    {this.getTransportIcons()}
                    {upperFirst(textForPast)}
                </span>

                <div />

                <div className={b('toggle')} onClick={this.onClickToggle}>
                    {segmentTransferKeyset('route')}
                    <Icon
                        glyph={IconGlyph.arrowUpBold}
                        className={b('toggleIcon')}
                    />
                </div>
            </div>
        );
    }

    getScheduleAndPrices(withTravelTrainTransferButton) {
        const {seoQueryParams, currencies, language, flags, data} = this.props;
        const {context, sort, segment} = data;

        const {isGone, queryingPrices} = segment;

        const {mainTariffKey: tariffClass, tariffKeys: tariffClassKeys} =
            getTariffKeys(segment, flags, seoQueryParams, sort);

        const defaultOpened = this.defaultOpened();
        const canShowOnlyDirectTrainVariantAction =
            checkShowOnlyDirectTrainVariantActions(flags);

        if (
            withTravelTrainTransferButton &&
            !canShowOnlyDirectTrainVariantAction
        ) {
            return (
                <PriceAndActionButton
                    className={b('mainPriceAndAction')}
                    subSegments={segment.segments}
                    context={context}
                />
            );
        }

        if (tariffClass || queryingPrices || defaultOpened) {
            // Интерлайн или пересадки раскрыты по-умолчанию и нет смысла в дополнительной кнопке-открывашке
            return (
                <ScheduleAndPrices
                    b={b}
                    segment={segment}
                    context={context}
                    language={language}
                    currencies={currencies}
                    queryingPrices={queryingPrices}
                    tariffClassKeys={tariffClassKeys}
                    klass={tariffClass}
                    isGone={isGone}
                />
            );
        }

        const segmentsQueryingPrices = segment.segments.some(
            transferSegment => transferSegment.queryingPrices,
        );

        if (segmentsQueryingPrices) {
            // Происходит полин цен сегментов пересадки
            return (
                <div className={b('scheduleAndPrices')}>
                    <Spinner className={b('priceSpinner')} size="tiny" />
                </div>
            );
        }

        const segmentsHavePrice = segment.segments.some(transferSegment =>
            havePriceAndSeats(transferSegment),
        );

        if (segmentsHavePrice && !canShowOnlyDirectTrainVariantAction) {
            // Отображаем кнопку-открывашку
            return (
                <div className={b('scheduleAndPrices')}>
                    <span
                        className={b('toggleBuyButton')}
                        onClick={this.onClickToggle}
                    >
                        {segmentTransferKeyset('toggle-show-segments')}
                        <Icon
                            glyph={IconGlyph.arrowUpBold}
                            className={b('toggleBuyButtonIcon')}
                        />
                    </span>
                </div>
            );
        }
    }

    render() {
        const {data, className, flags} = this.props;
        const {
            context,
            isHidden,
            showArrivalDate = false,
            showDepartureDate = false,
            segment,
        } = data;

        const {isGone, isInterval, segments, badges} = segment;

        const withTravelTrainTransferButton =
            needToShowTravelTrainTransferButton(segments);
        const canShowOnlyDirectTrainVariantAction =
            checkShowOnlyDirectTrainVariantActions(flags);

        const isOpened = this.isOpened();

        const mods = {
            isInterval,
            isGone,
            isHidden,
            isOpened,
            isNotEmptyBadges: !isEmpty(badges),
        };

        return (
            <tr className={b(mods, className)}>
                <td className={b('common')}>
                    {this.getThread()}

                    <TimeAndDuration
                        context={context}
                        segment={segment}
                        showDepartureDate={showDepartureDate}
                        showArrivalDate={showArrivalDate}
                    />

                    {this.getScheduleAndPrices(withTravelTrainTransferButton)}
                </td>

                <Segments
                    b={b}
                    data={data}
                    opened={isOpened}
                    segments={segments}
                    withoutPrices={canShowOnlyDirectTrainVariantAction}
                    withActionButton={
                        !withTravelTrainTransferButton &&
                        !canShowOnlyDirectTrainVariantAction
                    }
                />

                <td className={b('badges')}>
                    <SegmentBadges badges={badges} />
                </td>
            </tr>
        );
    }
}

export default connect(mapStateToProps)(TableSearchTransfer);
