import {React, B} from '../base';

import {useEffect, useCallback, ReactElement, useMemo} from 'react';
import moment from 'moment';
import upperFirst from 'lodash/upperFirst';

import ITransferSegment from '../../interfaces/transfer/ITransferSegment';
import ITransfer from '../../interfaces/transfer/ITransfer';
import ISearchSegment from '../../interfaces/components/ISearchSegment';
import IconGlyph from '../../interfaces/components/IconGlyph';

import {reachGoalOnce} from '../../lib/yaMetrika';
import {getTransferPointDescription} from '../../lib/segments/transfer/transferPoint';
import {makeCacheable} from '../../lib/cache';
import getEndpointsData from '../../lib/segments/endpointEvents/getEndpointsData';
import {humanizeDuration} from '../../lib/date/formats';
import {
    getLocalDepartureTime,
    getLocalArrivalTime,
} from '../../lib/segments/cachedSegmentTime';
import isAllDaysSearch from '../../lib/search/isAllDaysSearch';
import checkIsInterline from '../../lib/segments/checkIsInterline';
import havePriceAndSeats from '../../lib/segments/tariffs/havePriceAndSeats';
import getLowestPrice from '../../lib/segments/getLowestPrice';
import convertPrice from '../../lib/currencies/convertPrice';
import getMinTotalPrice from './components/PriceAndActionButton/utilities/getMinTotalPrice';
import needToShowTravelTrainTransferButton from './utilities/needToShowTravelTrainTransferButton';
import useDispatch from '../useDispatch';
import useSelector from '../useSelector';

import {setSegmentIsOpened} from '../../actions/search';

import SearchSegment from '../SearchSegment/SearchSegment.mobile';
import SegmentLabel from '../SearchSegment/components/SegmentLabel/SegmentLabel';
import TransportIcon from '../TransportIcon/TransportIcon';
import SegmentTimeAndStations from '../SegmentTimeAndStations/SegmentTimeAndStations.mobile';
import SearchPricesBlock from '../SearchPricesBlock';
import Icon from '../Icon/Icon';
import GlobalPopup from '../GlobalPopup';
import Arrow from '../Arrow/Arrow';
import Price from '../Price/Price';
import ModalHistoryWrapper from '../ModalHistoryWrapper/ModalHistoryWrapper';
import Spinner from '../Spinner/Spinner';
import ActionButton from './components/PriceAndActionButton/components/ActionButton/ActionButton';

import segmentTransferKeyset from '../../i18n/segment-transfer';
import checkShowOnlyDirectTrainVariantActions from './utilities/checkShowOnlyDirectTrainVariantActions';

const b = B('SearchTransfer');

// Приходится костылить, из-за формы данных в common/components/SearchSegments/SearchSegments, иначе будут лишние перерендеры сегментов пересадок
const getDataForTransferSegment = makeCacheable(
    (
        {data}: ISearchSegment<ITransfer>,
        segment: ITransferSegment,
    ): ISearchSegment<ITransferSegment> => {
        return {
            data: {
                ...data,
                segment,
                showDepartureDate: true,
                showArrivalDate: true,
                isStationsVisible: true,
            },
        };
    },
);

export default React.memo(SearchTransfer);

function SearchTransfer(
    params: ISearchSegment<ITransfer>,
): ReactElement | null {
    const {data, className} = params;

    const {
        segment: transfer,
        context,
        currentTimeMoment,
        isHidden,
        sort,
        flags,
        showArrivalDate = false,
        showDepartureDate = false,
    } = data;

    const {
        segments,
        badges,
        id,
        title,
        transport,
        departureMoment = getLocalDepartureTime(transfer),
        arrivalMoment = getLocalArrivalTime(transfer),
        departureRailwayMoment,
        arrivalRailwayMoment,
        stationFrom,
        stationTo,
        duration,
        isOpened,
        isGone,
    } = transfer;

    const dispatch = useDispatch();

    const onClickToggle = useCallback(() => {
        if (!isOpened) {
            reachGoalOnce('transfer_show_route');
        }

        dispatch(setSegmentIsOpened({id, isOpened: !isOpened}));
    }, [dispatch, id, isOpened]);

    const isInterline = checkIsInterline(transfer);
    const minTotalPrice = isInterline
        ? getLowestPrice(transfer)
        : getMinTotalPrice(segments);

    const withTravelTrainTransferButton = useMemo(
        () => needToShowTravelTrainTransferButton(segments),
        [segments],
    );
    const canShowOnlyDirectTrainVariantAction =
        checkShowOnlyDirectTrainVariantActions(flags);

    const actionButton = useMemo(() => {
        if (
            !withTravelTrainTransferButton ||
            canShowOnlyDirectTrainVariantAction
        ) {
            return null;
        }

        if (
            segments.some(
                subSegment =>
                    subSegment.queryingPrices &&
                    subSegment.tariffClassKeys.length === 0,
            )
        ) {
            return (
                <div className={b('spinnerWrapper')}>
                    <Spinner size="tiny" />
                </div>
            );
        }

        if (minTotalPrice) {
            return (
                <ActionButton
                    className={b('actionButton')}
                    subSegments={segments}
                    context={context}
                />
            );
        }

        return null;
    }, [
        context,
        minTotalPrice,
        segments,
        withTravelTrainTransferButton,
        canShowOnlyDirectTrainVariantAction,
    ]);

    useEffect(() => {
        reachGoalOnce('transfers_shown');
    }, []);

    const currencies = useSelector(state => state.currencies);

    if (isHidden) {
        return null;
    }

    const {code: transportType} = transport;
    const segmentsHavePrice = transfer.segments.some(transferSegment =>
        havePriceAndSeats(transferSegment),
    );

    const endpointData = getEndpointsData(transfer);

    return (
        <article className={b({isGone}, className)}>
            <div className={b('label')}>
                <SegmentLabel
                    className={b('labelBlock')}
                    isGone={isGone}
                    isInterval={false}
                    badges={badges}
                    endpointData={endpointData}
                    currentTimeMoment={currentTimeMoment}
                    departureMoment={departureMoment}
                />
            </div>

            <div className={b('title')}>
                {segments.map((segment, index) => (
                    <TransportIcon
                        key={index}
                        className={b('transportIcon')}
                        transportType={segment.transport.code}
                        isExpress={segment.thread && segment.thread.isExpress}
                        isAeroExpress={
                            segment.thread && segment.thread.isAeroExpress
                        }
                    />
                ))}

                {upperFirst(title)}
            </div>

            <SegmentTimeAndStations
                className={b('timeAndStations')}
                transportType={transportType}
                isExpress={false}
                isInterval={false}
                departureMoment={departureMoment}
                arrivalMoment={arrivalMoment}
                stationFrom={stationFrom}
                stationTo={stationTo}
                showArrivalDate={showArrivalDate}
                showDepartureDate={showDepartureDate}
                isAllDaysSearch={isAllDaysSearch(context)}
                duration={duration}
                departureRailwayMoment={departureRailwayMoment || undefined}
                arrivalRailwayMoment={arrivalRailwayMoment || undefined}
            />

            {minTotalPrice && !canShowOnlyDirectTrainVariantAction && (
                <div className={b('minPriceBlock')} onClick={onClickToggle}>
                    <div className={b('minPriceText')}>
                        {segmentTransferKeyset('price')}
                    </div>
                    <Price
                        className={b('minPrice')}
                        price={convertPrice(minTotalPrice, currencies)}
                        from
                    />
                </div>
            )}

            {actionButton}

            <div className={b('toggleContainer')}>
                <div className={b('toggle')} onClick={onClickToggle}>
                    {segmentTransferKeyset(
                        segmentsHavePrice
                            ? 'route-with-prices'
                            : 'route-with-no-prices',
                    )}
                    <Arrow direction="right" className={b('toggleIcon')} />
                </div>
            </div>

            {isOpened && (
                <ModalHistoryWrapper onClose={onClickToggle}>
                    <GlobalPopup>
                        <div className={b('transfersContainer', {isInterline})}>
                            <div
                                className={b('closeButton')}
                                onClick={onClickToggle}
                            >
                                <Icon
                                    className={b('backIcon')}
                                    glyph={IconGlyph.backArrow}
                                />
                                {segmentTransferKeyset('return-button')}
                            </div>
                            <div className={b('segments')}>
                                {segments.map((segment, index) => {
                                    const {data: dataSegment} =
                                        getDataForTransferSegment(
                                            params,
                                            segment,
                                        );

                                    return (
                                        <React.Fragment key={index}>
                                            <SearchSegment
                                                data={dataSegment}
                                                className={b('transferSegment')}
                                                withoutSegmentLabel
                                                withoutPrices={
                                                    isInterline ||
                                                    canShowOnlyDirectTrainVariantAction
                                                }
                                                withActionButton={
                                                    !withTravelTrainTransferButton &&
                                                    !canShowOnlyDirectTrainVariantAction
                                                }
                                            />

                                            {index < segments.length - 1 &&
                                                getTransferPoint(
                                                    segment,
                                                    segments[index + 1],
                                                )}
                                        </React.Fragment>
                                    );
                                })}
                                {isInterline && (
                                    <div className={b('segmentPrices')}>
                                        <SearchPricesBlock
                                            segment={transfer}
                                            context={context}
                                            sort={sort}
                                        />
                                    </div>
                                )}
                            </div>
                        </div>
                    </GlobalPopup>
                </ModalHistoryWrapper>
            )}
        </article>
    );
}

function getTransferPoint(
    segment: ITransferSegment,
    nextSegment: ITransferSegment,
): ReactElement {
    const segmentArrival =
        segment.arrivalMoment || getLocalArrivalTime(segment);
    const nextSegmentDeparture =
        nextSegment.departureMoment || getLocalDepartureTime(nextSegment);
    const duration = moment.duration(nextSegmentDeparture.diff(segmentArrival));
    const durationText = humanizeDuration(duration);

    return (
        <div className={b('transferPoint')}>
            <Icon className={b('transferIcon')} glyph={IconGlyph.transfer} />

            <span>
                <span className={b('transferText')}>
                    {getTransferPointDescription(segment, nextSegment)}
                </span>
                <span className={b('transferTime')}>{` ${durationText}`}</span>
            </span>
        </div>
    );
}
