import moment from 'moment-timezone';
import B from 'bem-cn-lite';
import React, {memo, FC, useContext, useMemo} from 'react';

import {ROBOT} from '../../../../../../lib/date/formats';

import {TransportType} from '../../../../../../lib/transportType';
import ITransferSegment from '../../../../../../interfaces/transfer/ITransferSegment';
import ISearchContext from '../../../../../../interfaces/state/search/ISearchContext';

import SegmentBuyButtonUtmContext from '../../../../../../contexts/SegmentBuyButtonUtmContext';

import useSelector from '../../../../../useSelector';

import applyUtm from '../../../../../../lib/url/applyUtm';
import getTrainUrl from '../../../../../../lib/url/getTrainUrl';
import getSegmentStringifyParams from './utilities/getSegmentStringifyParams';

import Link from '../../../../../Link';
import Platform from '../../../../../../interfaces/Platform';

import keyset from '../../../../../../../common/i18n/common';

interface IActionButtonProps {
    subSegments: ITransferSegment[];
    context: ISearchContext;

    className?: string;
}

const b = B('ActionButton');

const isMobile = process.env.PLATFORM === Platform.mobile;

const ActionButton: FC<IActionButtonProps> = props => {
    const {className, subSegments, context} = props;

    const {getUtmMedium} = useContext(SegmentBuyButtonUtmContext);

    const tld = useSelector(state => state.tld);
    const isProduction = useSelector(state => state.environment.production);
    const seoQueryParams = useSelector(state => state.seoQueryParams);
    const clientId = useSelector(state => state.user.clientId);

    const utmMedium = getUtmMedium(TransportType.train, true);

    const [firstSegment, secondSegment] = subSegments;

    const when = useMemo(() => {
        const firstSegmentDepartureMoment = moment.tz(
            firstSegment.departure,
            firstSegment.stationFrom.timezone,
        );

        return firstSegmentDepartureMoment.format(ROBOT);
    }, [firstSegment.departure, firstSegment.stationFrom.timezone]);

    const travelLink = useMemo(
        () =>
            applyUtm(
                getTrainUrl({
                    isProduction,
                    path: '/order/',
                    query: {
                        when,
                        fromId: context.from.key,
                        toId: context.to.key,
                        forward: `${getSegmentStringifyParams(
                            firstSegment,
                        )},${getSegmentStringifyParams(secondSegment)}`,
                        citizenship: tld,
                    },
                }),
                seoQueryParams,
                clientId,
                utmMedium,
            ),
        [
            tld,
            clientId,
            context.from.key,
            context.to.key,
            firstSegment,
            isProduction,
            secondSegment,
            seoQueryParams,
            utmMedium,
            when,
        ],
    );

    return (
        <Link
            className={b({isMobile, isDesktop: !isMobile}, className)}
            href={travelLink}
            target="_blank"
        >
            {keyset('select-places')}
        </Link>
    );
};

export default memo(ActionButton);
