import React, {ReactElement, memo, SyntheticEvent} from 'react';
import B from 'bem-cn-lite';
import {useSelector} from 'react-redux';

import IState from '../../interfaces/state/IState';
import ISegment from '../../interfaces/segment/ISegment';
import ITransfer from '../../interfaces/transfer/ITransfer';
import ITransferSegment from '../../interfaces/transfer/ITransferSegment';
import ISubSegment from '../../interfaces/segment/ISubSegment';
import ISegmentTariffClass from '../../interfaces/segment/ISegmentTariffClass';

import {useFlags} from '../../hooks/useFlags';
import {getBuyLinkText} from '../../lib/segments/snippetUtils';
import {shouldShowSuburbanTariffs} from '../../lib/segments/tariffs/tariffs';
import noop from '../../lib/noop';

import SegmentBuyLink from '../SegmentBuyLink';
import SuburbanTariffs from '../SuburbanTariffs/SuburbanTariffs';
import SegmentDatePicker from '../SegmentDatePicker/SegmentDatePicker';

const b = B('SegmentActionButton');

interface ISegmentActionButton {
    segment: ISegment | ITransfer | ITransferSegment | ISubSegment;
    tariffClass: ISegmentTariffClass | null;
    tariffClassKeys: string[];

    openPartnerPicker?: (e: SyntheticEvent) => void;
}

export default memo(SegmentActionButton);

function SegmentActionButton({
    segment,
    tariffClass,
    tariffClassKeys,
    openPartnerPicker = noop,
}: ISegmentActionButton): ReactElement | null {
    const context = useSelector((state: IState) => state.search.context);
    const language = useSelector((state: IState) => state.language);
    const currencies = useSelector((state: IState) => state.currencies);
    const flags = useFlags();

    if ('runDays' in segment && segment.runDays) {
        return (
            <SegmentDatePicker
                className={b('choiceDate')}
                context={context}
                segment={segment}
                language={language}
                currencies={currencies}
                flags={flags}
                tariffClassKeys={tariffClassKeys}
            />
        );
    }

    const {tariffs} = segment;

    if (shouldShowSuburbanTariffs(context, segment)) {
        return (
            <SuburbanTariffs
                className={b('suburbanTariffs')}
                tariffs={tariffs}
                currencies={currencies}
            />
        );
    }

    if (!tariffClass) {
        return null;
    }

    const {codeshareCompany} = tariffClass;
    const text = getBuyLinkText(segment);

    return (
        <SegmentBuyLink
            className={b('buyLink')}
            segment={segment}
            tariffClass={tariffClass}
            openPartnerPicker={openPartnerPicker}
        >
            {text}
            {codeshareCompany && (
                <p className={b('codeshareCompany')}>
                    {codeshareCompany.title}
                </p>
            )}
        </SegmentBuyLink>
    );
}
