import B from 'bem-cn-lite';
import {publish} from 'pubsub-js';
import React, {ReactElement, useState, useCallback} from 'react';
import moment from 'moment';

import {SEGMENTS} from '../../lib/constants/pubSubEvents';

import ITransfer from '../../interfaces/transfer/ITransfer';
import ITransferSegment from '../../interfaces/transfer/ITransferSegment';
import ISegment from '../../interfaces/segment/ISegment';
import ISearchContext from '../../interfaces/state/search/ISearchContext';
import ISegmentTariffClass from '../../interfaces/segment/ISegmentTariffClass';

import isAllDaysSearchFunc from '../../lib/search/isAllDaysSearch';
import {getBuyLinkText} from '../../lib/segments/snippetUtils';
import {reachGoalOnce} from '../../lib/yaMetrika';
import {useFlags} from '../../hooks/useFlags';
import useSelector from '../useSelector';

import SegmentBuyLink from '../SegmentBuyLink';
import SegmentCalendar from '../search/SegmentCalendar';

import datePickerKeyset from '../../i18n/segment-date-picker';

const b = B('SegmentActionButton');

interface ISegmentActionButton {
    segment: ISegment | ITransfer | ITransferSegment;
    tariffClassKeys: string[];
    context: ISearchContext;

    mainTariffKey?: ISegmentTariffClass;
    className?: string;
    onClickBuyButton?: React.ReactEventHandler;
}

export default React.memo(SegmentActionButton);

function SegmentActionButton({
    segment,
    tariffClassKeys,
    context,

    mainTariffKey,
    className,
    onClickBuyButton,
}: ISegmentActionButton): ReactElement | null {
    const [datePickerIsOpened, setDatePickerIsOpened] = useState(false);

    const currencies = useSelector(state => state.currencies);
    const flags = useFlags();
    const language = useSelector(state => state.language);

    const onDayClick = useCallback(() => {
        reachGoalOnce('segment_datepicker_click');
        setDatePickerIsOpened(false);
    }, []);

    const onDatePickerOpen = useCallback(() => {
        reachGoalOnce('segment_datepicker_show');
        setDatePickerIsOpened(true);
    }, []);

    const onDatePickerClose = useCallback(() => {
        setDatePickerIsOpened(false);
    }, []);

    const onClickByLinkWithVariants = useCallback(
        (e: React.SyntheticEvent) => {
            e.preventDefault();
            publish(SEGMENTS.SEGMENT_PARTNER_PICKER_CLICK, {
                segment,
                klass: mainTariffKey,
            });
        },
        [segment, mainTariffKey],
    );

    const {transport, tariffs} = segment;
    const {code: transportType} = transport;

    const runDays = 'runDays' in segment ? segment.runDays : undefined;
    const isAllDaysSearch = isAllDaysSearchFunc(context);

    const nowMoment = moment(context.time.now);

    if (isAllDaysSearch && runDays) {
        return (
            <div className={b({}, className)}>
                <span
                    className={b('button')}
                    onClick={onDatePickerOpen}
                    title={datePickerKeyset(`button-title-${transportType}`)}
                >
                    {datePickerKeyset('button-text')}
                </span>

                {datePickerIsOpened && (
                    <SegmentCalendar
                        segment={segment}
                        context={context}
                        tariffClassKeys={tariffClassKeys}
                        flags={flags}
                        language={language}
                        currencies={currencies}
                        nowMoment={nowMoment}
                        opened={datePickerIsOpened}
                        onClose={onDatePickerClose}
                        onChange={onDayClick}
                    />
                )}
            </div>
        );
    }

    if (!tariffs || !mainTariffKey) {
        return null;
    }

    return (
        <div className={b({}, className)}>
            <SegmentBuyLink
                className={b('link')}
                segment={segment}
                tariffClass={mainTariffKey}
                returnNullIfNotLink
                openPartnerPicker={onClickByLinkWithVariants}
                onClick={onClickBuyButton}
            >
                {getBuyLinkText(segment)}
            </SegmentBuyLink>
        </div>
    );
}
