import {PropTypes, refCallback} from '../base';

import React, {PureComponent} from 'react';
import B from 'bem-cn-lite';
import {connect} from 'react-redux';

import {YBUS} from '../../lib/segments/tariffSources';

import {TransportType} from '../../lib/transportType';
import HttpMethod from '../../interfaces/HttpMethod';

import noop from '../../lib/noop';
import {reachGoalOnce} from '../../lib/yaMetrika';
import getPurchaseLinkObject from '../../lib/segments/getPurchaseLinkObject';
import {addOrderGoalHandler} from '../../lib/metrika/createOrderGoalHandler';
import shouldUseLinkToTrains from '../../lib/order/shouldUseLinkToTrains';
import isTransferSegment from '../../lib/segments/isTransferSegment';

import SegmentBuyButtonUtmContext from '../../contexts/SegmentBuyButtonUtmContext';
import Link from '../Link';

const b = B('SegmentBuyLink');

const mapStateToProps = ({seoQueryParams, flags, user}) => ({
    seoQueryParams,
    flags,
    clientId: user.clientId,
});

class SegmentBuyLink extends PureComponent {
    static propTypes = {
        segment: PropTypes.object.isRequired,
        tariffClass: PropTypes.object.isRequired,

        onClick: PropTypes.func,
        onShow: PropTypes.func,
        medium: PropTypes.string,
        className: PropTypes.string,
        returnNullIfNotLink: PropTypes.bool,
        pageType: PropTypes.string,
        openPartnerPicker: PropTypes.func,
    };

    static defaultProps = {
        className: '',
        onClick: noop,
        onShow: noop,
        returnNullIfNotLink: false,
        medium: '',
        pageType: '',
        openPartnerPicker: noop,
    };

    componentDidMount() {
        const {segment, onShow, pageType} = this.props;

        onShow();

        if (segment.transport.code === TransportType.bus) {
            if (segment.source === YBUS) {
                reachGoalOnce(
                    `segment_ybus_button_shown${
                        pageType ? `_${pageType}` : ''
                    }`,
                );
            } else {
                reachGoalOnce(
                    `segment_partner_button_shown${
                        pageType ? `_${pageType}` : ''
                    }`,
                );
            }
        }
    }

    onClickGET = e => {
        const {segment, onClick, pageType} = this.props;
        const onClickHandler = addOrderGoalHandler(
            onClick,
            segment.transport.code,
            segment.source,
            !shouldUseLinkToTrains(segment),
            segment.badges,
            pageType,
            segment.isTransfer,
            segment.isTransferSegment,
        );

        onClickHandler(e);
    };

    onClickPOST = e => {
        this.onClickGET(e);

        if (!e.defaultPrevented) {
            this._form.submit();
        }
    };

    static contextType = SegmentBuyButtonUtmContext;

    render() {
        const {
            className,
            segment,
            tariffClass,
            returnNullIfNotLink,
            openPartnerPicker,
            seoQueryParams,
            clientId,
            children,
        } = this.props;
        const {getUtmMedium} = this.context;

        const {variants} = tariffClass;
        const {
            transport: {code: transportCode},
        } = segment;

        const linkObject = getPurchaseLinkObject({
            segment,
            tariffClass,
            medium: getUtmMedium(transportCode, isTransferSegment(segment)),
            seoQueryParams,
            clientId,
        });

        if (!linkObject.href && returnNullIfNotLink) {
            return null;
        }

        const isGET = linkObject.method === HttpMethod.get;

        let onClickHandler;
        let href;

        if (variants) {
            href = null;
            onClickHandler = openPartnerPicker;
        } else {
            href = isGET ? linkObject.href : null;
            onClickHandler = isGET ? this.onClickGET : this.onClickPOST;
        }

        return (
            <Link
                className={b(undefined, className)}
                href={href}
                rel="nofollow"
                target="_blank"
                onClick={onClickHandler}
                colors="inherit"
            >
                {children}
                {!isGET && (
                    <form
                        key="form"
                        style={{display: 'none'}}
                        ref={refCallback(this, '_form')}
                        target="_blank"
                        method="post"
                        action={linkObject.href}
                    >
                        {Object.entries(linkObject.POSTParams).map(
                            ([name, value]) => (
                                <input
                                    type="text"
                                    key={name}
                                    name={name}
                                    value={value}
                                />
                            ),
                        )}
                    </form>
                )}
            </Link>
        );
    }
}

export default connect(mapStateToProps)(SegmentBuyLink);
