import {React, PureComponent, mix, bindMethods} from '../base';
import moment from 'moment';

import {MIN_MONTHS_TO_FULL_CALENDAR} from '../../lib/date/values';

import SecondaryPosition from '../../interfaces/lib/dimensions/SecondaryPosition';
import PrimaryPosition from '../../interfaces/lib/dimensions/PrimaryPosition';

import isCppk from '../../lib/segments/isCppk';
import {
    getToday,
    getRange,
    getOrderRange,
    extendRangeToArray,
    getInactiveMonths,
} from '../../lib/date/utils';
import {getMonthsByMask} from '../../lib/date/mask';
import {reachGoalOnce} from '../../lib/yaMetrika';
import {getAllDaysLinkComponent} from '../../lib/search/getAllDaysLinkComponent';

import Dropdown from '../Dropdown/Dropdown';
import CalendarScroller from '../CalendarScroller/CalendarScroller';

import keyset from '../../i18n/segment-date-picker';

const POPUP_POSITIONS = [
    [PrimaryPosition.bottom, SecondaryPosition.right],
    [PrimaryPosition.left, SecondaryPosition.center],
    [PrimaryPosition.top, SecondaryPosition.right],
];

const b = 'SegmentDatePicker';

export default class SegmentDatePicker extends PureComponent {
    constructor(props) {
        super(props);
        bindMethods(this, ['getContent']);
    }

    onDropdownOpen() {
        reachGoalOnce('segment_datepicker_show');
    }

    onDayClick() {
        reachGoalOnce('segment_datepicker_click');
    }

    getContent() {
        const {
            segment,
            context,

            flags,
            language,
            currencies,

            tariffClassKeys,
        } = this.props;

        const {time} = context;
        const {timezone} = time;

        const today = getToday(time);
        const range = getRange(today);
        const orderRange = getOrderRange(
            today,
            segment.transport.code,
            segment.salesLimitInDays,
            isCppk(segment),
        );
        const activeMonths = getMonthsByMask({mask: segment.runDays, timezone});
        const showAllMonths =
            activeMonths.length >= MIN_MONTHS_TO_FULL_CALENDAR;
        const months = showAllMonths
            ? extendRangeToArray(range, 'month', 'month')
            : activeMonths;
        const inactiveMonths = showAllMonths
            ? getInactiveMonths(months, activeMonths)
            : [];
        const nowMoment = moment(time.now);

        const DayComponent = getAllDaysLinkComponent({
            transportType: segment.transport.code,
            trainPurchaseNumbers: segment.trainPurchaseNumbers,
        });

        return (
            <CalendarScroller
                today={today}
                months={months}
                inactiveMonths={inactiveMonths}
                orderRange={orderRange}
                onClick={this.onDayClick}
                segment={segment}
                context={context}
                language={language}
                currencies={currencies}
                flags={flags}
                tariffClassKeys={tariffClassKeys}
                nowMoment={nowMoment}
                DayComponent={DayComponent}
            />
        );
    }

    render() {
        const {className, segment} = this.props;

        return (
            <Dropdown
                className={mix(b, className)}
                noArrow
                onOpen={this.onDropdownOpen}
                buttonText={keyset('button-text')}
                popupPositions={POPUP_POSITIONS}
                title={keyset(`button-title-${segment.transport.code}`)}
                getContent={this.getContent}
            />
        );
    }
}
