import {React, PureComponent, B} from '../base';

import {ReactElement} from 'react';
import {connect} from 'react-redux';

import {SUBURBAN} from '../../lib/segments/tariffClasses';

// eslint-disable-next-line no-duplicate-imports
import {TransportType} from '../../lib/transportType';
import ISegment from '../../interfaces/segment/ISegment';
import IStateCurrencies from '../../interfaces/state/IStateCurrencies';
import MapStateToProps from '../../interfaces/MapStateToProps';
import ITransferSegment from '../../interfaces/transfer/ITransferSegment';
import ITransfer from '../../interfaces/transfer/ITransfer';

import insertMarkupIntoKey from '../../i18n/_utils/insertMarkupIntoKey';
import getLowestPrice from '../../lib/segments/getLowestPrice';

import Price from '../Price/Price';
import Spinner from '../Spinner/Spinner';
import TariffsList from '../TariffsList/TariffsList';

import priceKeyset from '../../i18n/price';

const b = B('SegmentPrices');

interface ISegmentPricesFromState {
    currencies: IStateCurrencies;
}

interface ISegmentPricesProps extends ISegmentPricesFromState {
    segment: ISegment | ITransferSegment | ITransfer;
    tariffClassKeys: string[];
    isAllDaysSearch: boolean;
}

const mapStateToProps: MapStateToProps<ISegmentPricesFromState> = ({
    currencies,
}) => ({currencies});

class SegmentPrices extends PureComponent<ISegmentPricesProps, {}> {
    getLowestPriceBlock(): ReactElement | null {
        const {segment} = this.props;
        const {
            transport: {code: transportType},
        } = segment;

        const cheapestPrice = getLowestPrice(segment);

        if (!cheapestPrice) {
            return null;
        }

        const price = <Price key="cheapestPrice" price={cheapestPrice} />;

        return (
            <div className={b('cheapestPriceWrapper')}>
                {transportType !== TransportType.suburban
                    ? insertMarkupIntoKey(priceKeyset, 'from', {price})
                    : price}
            </div>
        );
    }

    getAllDaysSearchMarkup(): ReactElement | null {
        const {segment} = this.props;
        const {
            transport: {code: transportType},
            tariffClassKeys,
        } = segment;

        if (
            transportType === TransportType.suburban &&
            tariffClassKeys.length === 1 &&
            tariffClassKeys[0] === SUBURBAN
        ) {
            // Если это электричка и у нее нет поездатых цен, то ее цены статичны и не меняются и
            // можно отобразить их так же как и на конкретный день.
            // Кроме того только так будут работать тарифы ЦППК, потому что нужно выводить именно несколько конкретных цен,
            // а не одну минимальную.
            return this.getDateSearchMarkup();
        }

        return this.getLowestPriceBlock();
    }

    getDateSearchMarkup(): ReactElement | null {
        const {segment, currencies, tariffClassKeys} = this.props;

        const {code: transportType} = segment.transport;

        if (transportType === TransportType.bus) {
            return this.getLowestPriceBlock();
        }

        if (!tariffClassKeys.length) {
            return null;
        }

        return (
            <div className={b('scrollableWrapper')}>
                <div className={b('scrollable')}>
                    <div className={b('scrollableContent')}>
                        <TariffsList
                            segment={segment}
                            tariffClassKeys={tariffClassKeys}
                            currencies={currencies}
                        />
                    </div>
                </div>
            </div>
        );
    }

    getContent(): ReactElement | null {
        const {segment, tariffClassKeys, isAllDaysSearch} = this.props;
        const {queryingPrices} = segment;

        if (tariffClassKeys.length === 0 && queryingPrices) {
            return <Spinner className={b('priceSpinner')} size="tiny" />;
        }

        if (isAllDaysSearch) {
            return this.getAllDaysSearchMarkup();
        }

        return this.getDateSearchMarkup();
    }

    render(): ReactElement | null {
        const content = this.getContent();

        if (!content) {
            return null;
        }

        return <div className={b()}>{content}</div>;
    }
}

export default connect(mapStateToProps)(SegmentPrices);
