import {React, B} from '../base';

import moment, {Moment} from '../../../reexports/moment-timezone';

import {HUMAN_SHORT, TIME} from '../../lib/date/formats';
import {HOURS} from '../../lib/date/constants';
import {CHAR_NBSP, CHAR_EM_DASH} from '../../lib/stringUtils';

import {TransportType} from '../../lib/transportType';
import ISegmentStation from '../../interfaces/segment/ISegmentStation';
import ITransferStation from '../../interfaces/transfer/ITransferStation';
import IPoint from '../../interfaces/state/searchForm/IPoint';
import Lang from '../../interfaces/Lang';
import Tld from '../../interfaces/Tld';
import ExpValues from '../../interfaces/state/flags/ExpValues';

import useSelector from '../useSelector';
import {useFlags} from '../../hooks/useFlags';
import {formatLocalTime} from '../../lib/date/utils';
import {stationUrl} from '../../lib/url/stationUrl';
// eslint-disable-next-line no-duplicate-imports
import {humanizeDuration} from '../../lib/date/formats';
import getRailwayTime from '../../lib/date/getRailwayTime';

import Link from '../Link';

import commonKeyset from '../../i18n/common';
import timeKeyset from '../../i18n/time';
import getSubtypeByTransportType from '../../lib/station/getSubtypeByTransportType';

const b = B('TimeAndStations');

const NO_DATA_TEXT_FOR_TIME = '-- --';

type Station = ISegmentStation | ITransferStation;

interface ITimeAndStations {
    transportType: TransportType;
    isExpress: boolean;
    isInterval: boolean;
    departureMoment: Moment;
    arrivalMoment: Moment;
    stationFrom: Station;
    stationTo: Station;
    showArrivalDate: boolean;
    showDepartureDate: boolean;
    isAllDaysSearch: boolean;

    duration?: number; // Время в пути, в секундах
    beginTime?: string;
    endTime?: string;
    maxArrival?: string;
    minArrival?: string;
    isFuzzyFrom?: boolean;
    isFuzzyTo?: boolean;
    useCityInsteadStationTo?: boolean;
    useCityInsteadStationFrom?: boolean;
    cityFrom?: IPoint;
    cityTo?: IPoint;
    departureRailwayMoment?: Moment;
    arrivalRailwayMoment?: Moment;
    className?: string;
}

export default React.memo(TimeAndStations);

function TimeAndStations({
    transportType,
    isExpress,
    isInterval,
    departureMoment,
    arrivalMoment,
    stationFrom,
    stationTo,
    showArrivalDate,
    showDepartureDate,
    isAllDaysSearch,

    duration,
    beginTime,
    endTime,
    maxArrival,
    minArrival,
    isFuzzyFrom,
    isFuzzyTo,
    useCityInsteadStationTo = false,
    useCityInsteadStationFrom = false,
    cityFrom,
    cityTo,
    departureRailwayMoment,
    arrivalRailwayMoment,
    className,
}: ITimeAndStations): React.ReactElement {
    const language = useSelector(state => state.language);
    const tld = useSelector(state => state.tld);

    const user = useSelector(state => state.user);
    const flags = useFlags();

    if (isInterval && beginTime && endTime) {
        return intervalTimeAndStations(
            transportType,
            beginTime,
            endTime,
            stationFrom,
            stationTo,
            language,
            tld,

            duration,
            useCityInsteadStationTo,
            useCityInsteadStationFrom,
            cityFrom,
            cityTo,
            className,
        );
    }

    const isRangeArrival = Boolean(
        maxArrival && minArrival && maxArrival !== minArrival,
    );
    const noDataText = commonKeyset('no-data');
    const minArrivalMoment = isRangeArrival
        ? moment.tz(minArrival, stationTo.timezone)
        : arrivalMoment;
    const maxArrivalMoment = isRangeArrival
        ? moment.tz(maxArrival, stationTo.timezone)
        : undefined;
    const durationTime = moment.duration(
        minArrivalMoment.diff(departureMoment),
    );
    const humanDuration = humanizeDuration(
        durationTime,
        isRangeArrival ? {shortestUnit: HOURS} : {},
    );
    const durationText = `${
        isRangeArrival ? timeKeyset('duration-from') : ''
    } ${humanDuration}`.trim();
    const departureIsValid = departureMoment.isValid();
    const minArrivalIsValid = minArrivalMoment.isValid();
    const maxArrivalIsValid = maxArrivalMoment
        ? maxArrivalMoment.isValid()
        : false;
    const departureRailwayTime = isRangeArrival
        ? null
        : getRailwayTime({
              isAllDaysSearch,
              timeMoment: departureMoment,
              railwayMoment: departureRailwayMoment,
              railwayTimezone: stationFrom && stationFrom.railwayTimezone,
          });
    const arrivalRailwayTime = isRangeArrival
        ? null
        : getRailwayTime({
              isAllDaysSearch,
              timeMoment: minArrivalMoment,
              railwayMoment: arrivalRailwayMoment,
              railwayTimezone: stationTo && stationTo.railwayTimezone,
          });

    const showDateRow =
        flags.hideMoscowTime !== ExpValues.experiment && !user.isBot;

    return (
        <table
            className={b(
                {
                    type: transportType,
                    isExpress,
                    isRangeArrival,
                },
                className,
            )}
        >
            <tbody>
                <tr className={b('dateRow')}>
                    <td className={b('departureDate')} data-nosnippet>
                        {showDepartureDate &&
                            (departureIsValid
                                ? departureMoment
                                      .format(HUMAN_SHORT)
                                      .replace(/\.$/, '')
                                : noDataText)}
                        {showDepartureDate &&
                            departureRailwayTime &&
                            showDateRow &&
                            ' | '}
                        {departureRailwayTime && showDateRow && (
                            <span
                                className="TooltipCss"
                                aria-label={departureRailwayTime.title}
                            >
                                {departureRailwayTime.time.replace(/\.$/, '')}
                            </span>
                        )}
                    </td>

                    <td className={b('arrivalDate')} data-nosnippet>
                        {showArrivalDate &&
                            (minArrivalIsValid
                                ? minArrivalMoment
                                      .format(HUMAN_SHORT)
                                      .replace(/\.$/, '')
                                : noDataText)}
                        {showArrivalDate &&
                            arrivalRailwayTime &&
                            showDateRow &&
                            ' | '}
                        {arrivalRailwayTime && showDateRow && (
                            <span
                                className="TooltipCss"
                                aria-label={arrivalRailwayTime.title}
                            >
                                {arrivalRailwayTime.time.replace(/\.$/, '')}
                            </span>
                        )}
                    </td>

                    <td className={b('arrivalDateMax')}>
                        {maxArrivalMoment &&
                            showArrivalDate &&
                            (maxArrivalIsValid
                                ? maxArrivalMoment
                                      .format(HUMAN_SHORT)
                                      .replace(/\.$/, '')
                                : noDataText)}
                    </td>
                </tr>

                <tr className={b('timeRow')}>
                    <td className={b('departureTimeContainer')}>
                        <div
                            className={b(
                                'departureTime',
                                !departureIsValid || isFuzzyFrom
                                    ? 'TooltipCss'
                                    : '',
                            )}
                            aria-label={
                                isFuzzyFrom
                                    ? timeKeyset('approximate')
                                    : departureIsValid
                                    ? ''
                                    : noDataText
                            }
                        >
                            {isFuzzyFrom &&
                                getFuzzyPrefix(b('fuzzyDepartureTime'))}
                            {departureIsValid
                                ? departureMoment.format(TIME)
                                : NO_DATA_TEXT_FOR_TIME}
                        </div>
                        <div className={b('line')} />
                        <div className={b('duration')}>{durationText}</div>
                        <div className={b('line')} />
                        {isFuzzyTo && getFuzzyPrefix(b('fuzzyArrivalTime'))}
                    </td>

                    <td
                        className={b(
                            'arrivalTime',
                            !minArrivalIsValid || isFuzzyTo ? 'TooltipCss' : '',
                        )}
                        aria-label={
                            isFuzzyTo
                                ? timeKeyset('approximate')
                                : minArrivalIsValid
                                ? ''
                                : noDataText
                        }
                    >
                        {minArrivalIsValid
                            ? minArrivalMoment.format(TIME)
                            : NO_DATA_TEXT_FOR_TIME}
                        {maxArrivalMoment && '...'}
                    </td>

                    <td
                        className={b(
                            'arrivalTimeMax',
                            !maxArrivalMoment
                                ? ''
                                : maxArrivalIsValid
                                ? ''
                                : 'TooltipCss',
                        )}
                        aria-label={
                            !maxArrivalMoment
                                ? ''
                                : maxArrivalIsValid
                                ? ''
                                : noDataText
                        }
                    >
                        {maxArrivalMoment &&
                            (maxArrivalIsValid
                                ? maxArrivalMoment.format(TIME)
                                : NO_DATA_TEXT_FOR_TIME)}
                    </td>
                </tr>

                <tr className={b('stationRow')}>
                    <td className={b('departureStation')}>
                        {getStation(
                            stationFrom,
                            useCityInsteadStationFrom,
                            transportType,
                            tld,
                            language,
                            cityFrom,
                        )}
                    </td>

                    <td className={b('arrivalStation')} colSpan={2}>
                        {getStation(
                            stationTo,
                            useCityInsteadStationTo,
                            transportType,
                            tld,
                            language,
                            cityTo,
                        )}
                    </td>
                </tr>

                <tr className={b('platformRow')}>
                    <td className={b('departurePlatform')}>
                        {'platform' in stationFrom
                            ? stationFrom.platform
                            : null}
                    </td>

                    <td className={b('arrivalPlatform')} colSpan={2}>
                        {'platform' in stationTo ? stationTo.platform : null}
                    </td>
                </tr>
            </tbody>
        </table>
    );
}

function intervalTimeAndStations(
    transportType: TransportType,
    beginTime: string,
    endTime: string,
    stationFrom: Station,
    stationTo: Station,
    language: Lang,
    tld: Tld,

    duration?: number, // время в пути, в секундах
    useCityInsteadStationTo?: boolean,
    useCityInsteadStationFrom?: boolean,
    cityFrom?: IPoint,
    cityTo?: IPoint,
    className?: string,
): React.ReactElement {
    const humanDuration =
        duration && humanizeDuration(moment.duration(duration, 's'));

    return (
        <div className={b({transportType, isInterval: true}, className)}>
            <div className={b('intervalTimesContainer')}>
                <span className={b('intervalTimes')}>
                    {timeKeyset('interval', {
                        start: formatLocalTime(beginTime),
                        end: formatLocalTime(endTime),
                    })}
                </span>
                <span className={b('intervalDuration')}>{humanDuration}</span>
            </div>
            <div className={b('intervalStations')}>
                {getStation(
                    stationFrom,
                    useCityInsteadStationFrom || false,
                    transportType,
                    tld,
                    language,
                    cityFrom,
                )}
                {`${CHAR_NBSP}${CHAR_EM_DASH} `}
                {getStation(
                    stationTo,
                    useCityInsteadStationTo || false,
                    transportType,
                    tld,
                    language,
                    cityTo,
                )}
            </div>
        </div>
    );
}

function getStation(
    station: ISegmentStation | ITransferStation,
    useCityInsteadStation: boolean,
    transportType: TransportType,
    tld: Tld,
    language: Lang,

    city?: IPoint,
): React.ReactNode {
    if (useCityInsteadStation && city) {
        return city.popularTitle || city.title;
    }

    // Если не изветсен mainSubtype, то не передаем subtype, потому что некотрые ручки тарифов
    // ничего не знают про mainSubtype и если добавлять subtype, то может быть 302-редирект на странице станции
    const subtype = station.mainSubtype
        ? getSubtypeByTransportType(transportType)
        : undefined;

    return (
        <Link
            href={stationUrl({
                id: station.id,
                type: station.pageType,
                mainSubtype: station.mainSubtype,
                subtype,
                isMobile: true,
                tld,
                language,
            })}
            colors="inherit"
        >
            {station.popularTitle || station.title}
        </Link>
    );
}

function getFuzzyPrefix(className?: string): React.ReactElement {
    return (
        <div
            className={b('fuzzyTime', `TooltipCss ${className}`)}
            aria-label={timeKeyset('approximate')}
        >
            ~
        </div>
    );
}
