import React, {
    ReactElement,
    useCallback,
    memo,
    useContext,
    useMemo,
    useRef,
    useState,
    RefObject,
    useEffect,
} from 'react';
import B from 'bem-cn-lite';
import throttle from 'lodash/throttle';

import {CHAR_NBSP} from '../../lib/stringUtils';

import ISegment from '../../interfaces/segment/ISegment';
import ISearchContext from '../../interfaces/state/search/ISearchContext';

import getTransportFeatures from '../../lib/segments/transportFeatures/getTransportFeatures';
import patchTitle from '../../lib/segments/patchTitle';
import getThreadUrlForSegment from '../../lib/segments/getThreadUrlForSegment';
import {isCity} from '../../lib/point/pointType';
import {useFlags} from '../../hooks/useFlags';
import useSelector from '../useSelector';
import useDispatch from '../useDispatch';

import {setSegmentCodesharesIsOpened} from '../../actions/search';

import SegmentTitleLinkUtmContext from '../../contexts/SegmentTitleLinkUtmContext';
import Link from '../Link';
import TransportIcon from '../TransportIcon/TransportIcon';
import SegmentTransport from '../SegmentTransport/SegmentTransport';
import MoreButton from '../MoreButton/MoreButton';

import threadKeyset from '../../i18n/thread';
import segmentKeyset from '../../i18n/segment';

const b = B('SegmentTitle');

interface ISegmentTitle {
    segment: ISegment;
    context: ISearchContext;
    showTransportIcon: boolean;

    className?: string;
    isSuburbanSearchResult?: boolean;
}

export default memo(SegmentTitle);

function SegmentTitle({
    segment,
    context,
    showTransportIcon,

    className,
    isSuburbanSearchResult,
}: ISegmentTitle): ReactElement {
    const {
        transport: {code: transportType},
        codeshares,
        codesharesIsOpened,
    } = segment;

    const language = useSelector(state => state.language);
    const tld = useSelector(state => state.tld);
    const flags = useFlags();
    const seoQueryParams = useSelector(state => state.seoQueryParams);
    const clientId = useSelector(state => state.user.clientId);
    const isProduction = useSelector(state => state.environment.production);

    const dispatch = useDispatch();
    const {getUtmMedium} = useContext(SegmentTitleLinkUtmContext);

    const fakeCodesharesListRef = useRef<HTMLSpanElement>(null);
    const codesharesContainerRef = useRef<HTMLDivElement>(null);

    const onCodesharesMoreButtonClick = useCallback(() => {
        const {id} = segment;

        dispatch(setSegmentCodesharesIsOpened({id, codesharesIsOpened: true}));
    }, [dispatch, segment]);

    const header = useMemo(() => {
        const {title, number, thread, transport} = segment;

        const children: ReactElement[] = [];

        if (showTransportIcon) {
            children.push(
                <TransportIcon
                    key="icon"
                    isExpress={thread?.isExpress}
                    isAeroExpress={thread?.isAeroExpress}
                    transportType={transport.code}
                />,
            );
        }

        if (number && number.trim() !== '') {
            children.push(
                <span className={b('number')} key="number">
                    {`${number.replace(/\s/g, CHAR_NBSP)} `}
                </span>,
            );
        }

        children.push(
            <h3 className={b('title')} key="title">
                {patchTitle(title)}
            </h3>,
        );

        const link = getThreadUrlForSegment({
            segment,
            tld,
            language,
            seoQueryParams,
            clientId,
            utmMedium: getUtmMedium(transport.code),
            flags,
            isProduction,
            isToCitySearchContext: isCity(context.to),
        });

        const mainInfo = !link ? (
            children
        ) : (
            <Link
                className={b('link')}
                href={link}
                title={threadKeyset(`title-${transport.code}`, {number, title})}
            >
                {children}
            </Link>
        );

        return <div className={b('header')}>{mainInfo}</div>;
    }, [
        context,
        clientId,
        getUtmMedium,
        language,
        segment,
        seoQueryParams,
        showTransportIcon,
        tld,
        flags,
        isProduction,
    ]);

    const codesharesText =
        codeshares && codeshares.length > 0
            ? segmentKeyset('codeshares-list', {
                  text: codeshares
                      .map(({number, company}) => `${number} ${company.title}`)
                      .join(', '),
              })
            : '';

    const [codesharesContainerWidth, setCodesharesContanerWidth] = useState<
        number | undefined
    >(undefined);
    const [codesharesTextInfo, setCodesharesTextInfo] = useState<{
        forCodesharesText: string;
        width?: number;
    }>({
        forCodesharesText: codesharesText,
    });

    const calculateAndSetCodesharesContainerWidth = useCallback(() => {
        setCodesharesContanerWidth(getWidthByRef(codesharesContainerRef));
    }, []);

    // Считаем первоначальную ширину контейнера кодшера
    useEffect(() => {
        if (!codesharesText) {
            return;
        }

        if (!codesharesContainerWidth) {
            calculateAndSetCodesharesContainerWidth();
        }
    }, [
        calculateAndSetCodesharesContainerWidth,
        codesharesContainerWidth,
        codesharesText,
    ]);

    // Считаем ширину текста кодшеров
    useEffect(() => {
        if (!codesharesText) {
            return;
        }

        const {forCodesharesText, width} = codesharesTextInfo;

        if (forCodesharesText !== codesharesText || !width) {
            setCodesharesTextInfo({
                forCodesharesText: codesharesText,
                width: getWidthByRef(fakeCodesharesListRef),
            });
        }
    }, [codesharesText, codesharesTextInfo]);

    // Ставим обработчик на вычисление ширины контейнера кодшеров, при ресайзах окна
    useEffect(() => {
        if (!codesharesText) {
            return;
        }

        const throttledCalculateAndSetCodesharesContainerWidth = throttle(
            calculateAndSetCodesharesContainerWidth,
            200,
            {leading: false, trailing: true},
        );

        window.addEventListener(
            'resize',
            throttledCalculateAndSetCodesharesContainerWidth,
        );

        return () => {
            throttledCalculateAndSetCodesharesContainerWidth.cancel();
            window.removeEventListener(
                'resize',
                throttledCalculateAndSetCodesharesContainerWidth,
            );
        };
    }, [calculateAndSetCodesharesContainerWidth, codesharesText]);

    const showCodesharesMoreButton = Boolean(
        codesharesContainerWidth &&
            codesharesTextInfo.forCodesharesText === codesharesText &&
            codesharesTextInfo.width &&
            codesharesContainerWidth <= codesharesTextInfo.width,
    );

    return (
        <div className={b({codesharesIsOpened}, className)}>
            {header}

            <SegmentTransport
                className={b('transport')}
                features={getTransportFeatures(
                    segment,
                    false,
                    tld,
                    language,
                    isSuburbanSearchResult,
                )}
                transportType={transportType}
            />

            {codesharesText && (
                <>
                    <div
                        className={b('codeshares')}
                        ref={codesharesContainerRef}
                    >
                        <span className={b('codesharesList')}>
                            {codesharesText}
                        </span>

                        {showCodesharesMoreButton && (
                            <MoreButton
                                className={b('codesharesMoreButton')}
                                size="s"
                                onClick={onCodesharesMoreButtonClick}
                            />
                        )}
                    </div>

                    {/* Для рассчета длины текста */}
                    <div className={b('fakeCodeshares')}>
                        <span
                            className={b('fakeCodesharesList')}
                            ref={fakeCodesharesListRef}
                        >
                            {codesharesText}
                        </span>
                    </div>
                </>
            )}
        </div>
    );
}

function getWidthByRef(ref: RefObject<HTMLElement>): number | undefined {
    if (!ref.current) {
        return undefined;
    }

    return ref.current.offsetWidth;
}
