import {React, B} from '../base';

import {
    SyntheticEvent,
    ReactElement,
    useCallback,
    memo,
    Fragment,
    useRef,
    useContext,
} from 'react';
import page from 'page';
import {publish} from 'pubsub-js';

import {SEGMENTS} from '../../lib/constants/pubSubEvents';

import {TransportType} from '../../lib/transportType';
import ISegmentTransport from '../../interfaces/segment/ISegmentTransport';
import ISegmentCompany from '../../interfaces/segment/ISegmentCompany';
import ISegmentThreadBase from '../../interfaces/segment/ISegmentThreadBase';
import ISegmentFacility from '../../interfaces/segment/ISegmentFacility';
import ISegmentCodeshare from '../../interfaces/segment/ISegmentCodeshare';
import IconGlyph from '../../interfaces/components/IconGlyph';

import patchTitle from '../../lib/segments/patchTitle';
import getSubtype from '../../lib/segments/transportFeatures/getSubtype';
import getCompanies from '../../lib/segments/transportFeatures/getCompanies';
import hasElectronicRegistration from '../../lib/segments/transportFeatures/hasElectronicRegistration';
import getCrowdTestingUrl from '../../lib/url/getCrowdTestingUrl';
import useSelector from '../useSelector';

import TransportIcon from '../TransportIcon/TransportIcon';
import Link from '../Link';
import Icon from '../Icon/Icon';
import StaticContext from '../StaicContext/StaticContext';

import segmentKeyset from '../../i18n/segment';

const b = B('SegmentTitle');

interface ISegmentTitle {
    transport: ISegmentTransport;
    title: string;
    isSuburbanSearchResult: boolean;

    className?: string;
    number?: string;
    showTransportIcon?: boolean;
    thread?: ISegmentThreadBase;
    isThroughTrain?: boolean;
    company?: ISegmentCompany;
    companies?: ISegmentCompany[];
    electronicTicket?: boolean;
    hasDynamicPricing?: boolean;
    threadUrl?: string;
    facilities?: ISegmentFacility[];
    linkTitleColorsInherit?: boolean;
    showTitle?: boolean;
    codeshares?: ISegmentCodeshare[];
}

export default memo(SegmentTitle);

function SegmentTitle({
    transport,
    title,
    isSuburbanSearchResult,

    company,
    number,
    thread,
    className = '',
    showTransportIcon = true,
    isThroughTrain = false,
    companies,
    electronicTicket = false,
    hasDynamicPricing = false,
    threadUrl,
    facilities = [],
    linkTitleColorsInherit = false,
    showTitle = false,
    codeshares,
}: ISegmentTitle): ReactElement {
    const tld = useSelector(state => state.tld);
    const language = useSelector(state => state.language);
    const {isCrowdTesting} = useContext(StaticContext);

    const {code: transportType} = transport;
    const withCodeshares = Boolean(codeshares && codeshares.length);

    const infoContainerRef = useRef<HTMLSpanElement>(null);

    const onClick = useCallback(
        (e: SyntheticEvent) => {
            let url = threadUrl;

            if (!url) {
                return;
            }

            const {current} = infoContainerRef;
            const {target} = e;

            if (
                current === target ||
                (target instanceof Node && current?.contains(target))
            ) {
                return;
            }

            if (isCrowdTesting) {
                url = getCrowdTestingUrl(url);
            }

            page(url);
            e.preventDefault();
        },
        [threadUrl, isCrowdTesting],
    );

    const onClickInfo = useCallback(() => {
        const component = (
            <div className={b('sheetContent')}>
                <div className={b('sheetHeader')}>
                    {segmentKeyset('flight-with-codeshares-list', {
                        name: `${number} ${
                            company?.title || company?.shortTitle
                        }`,
                    })}
                </div>

                <table className={b('sheetTable')}>
                    <tbody>
                        {codeshares?.map(codeshare => (
                            <tr key={codeshare.number}>
                                <td className={b('codeshareCellNumber')}>
                                    {codeshare.number}
                                </td>
                                <td>
                                    {codeshare.company.title ||
                                        codeshare.company.shortTitle}
                                </td>
                            </tr>
                        ))}
                    </tbody>
                </table>
            </div>
        );

        publish(SEGMENTS.UNIVERSAL, {component});
    }, [codeshares, company, number]);

    let trainSubType: null | ReactElement = null;

    if (
        transportType === TransportType.train ||
        transportType === TransportType.suburban
    ) {
        const subtype = getSubtype(transport, thread, isSuburbanSearchResult);
        const subtypeName = subtype && subtype.content;
        const deluxeTrainName =
            thread && thread.deluxeTrain && thread.deluxeTrain.shortTitle;
        const isThroughTrainText: string = isThroughTrain
            ? segmentKeyset('through-train')
            : '';
        const trainTypeText =
            subtypeName || isThroughTrainText || deluxeTrainName;

        if (trainTypeText) {
            trainSubType = (
                <span className={b('trainType')}>{trainTypeText}</span>
            );
        }
    }

    const companiesArr = getCompanies(tld, language, company, companies);

    const header = (
        <Fragment>
            {Boolean(number && number.trim() !== '') && (
                <span className={b('number')}>{number}</span>
            )}

            {(showTitle ||
                (transportType !== TransportType.train &&
                    transportType !== TransportType.plane)) && (
                <span className={b('title')}>{patchTitle(title)}</span>
            )}

            {withCodeshares && companiesArr.length > 0 && (
                <span>
                    {' '}
                    {companiesArr.map(({content}) => content).join(', ')}
                </span>
            )}

            {trainSubType}
        </Fragment>
    );

    return (
        <div className={b({withCodeshares}, className)} onClick={onClick}>
            {showTransportIcon && (
                <TransportIcon
                    className={b('transportIcon')}
                    isExpress={Boolean(thread && thread.isExpress)}
                    isAeroExpress={Boolean(thread && thread.isAeroExpress)}
                    transportType={transportType}
                />
            )}

            {threadUrl ? (
                <Link
                    className={b('headerLink')}
                    href={threadUrl}
                    colors={linkTitleColorsInherit ? 'inherit' : 'travel'}
                >
                    {header}
                </Link>
            ) : (
                header
            )}

            {transportType !== TransportType.train &&
                transportType !== TransportType.suburban &&
                !withCodeshares &&
                Boolean(companiesArr.length) &&
                companiesArr.map(({content}, index) => (
                    <span className={b('company')} key={index}>
                        {content}
                    </span>
                ))}

            {withCodeshares && codeshares && (
                <>
                    <span className={b('codesharesContainer')}>
                        {codeshares.map(codeshare => (
                            <span
                                className={b('codeshare')}
                                key={codeshare.number}
                            >
                                {codeshare.number}
                            </span>
                        ))}
                    </span>

                    <span
                        className={b('infoContainer')}
                        ref={infoContainerRef}
                        onClick={onClickInfo}
                    >
                        <Icon
                            className={b('infoIcon')}
                            glyph={IconGlyph.info}
                        />
                    </span>
                </>
            )}

            {hasElectronicRegistration(transportType, electronicTicket) && (
                <span className={b('eTicket')}>
                    {segmentKeyset('e-ticket')}
                </span>
            )}

            {hasDynamicPricing && (
                <span className={b('dynamicPricing')}>
                    {segmentKeyset('dynamic-pricing')}
                </span>
            )}

            {isSuburbanSearchResult && Boolean(facilities.length) && (
                <span className={b('facilities')}>
                    {facilities.map(facility => (
                        <img
                            key={facility.icon}
                            className={b('facilityIcon')}
                            src={facility.icon}
                            alt={facility.title}
                        />
                    ))}
                </span>
            )}
        </div>
    );
}
