import React, {ReactElement, memo, RefObject} from 'react';
import B from 'bem-cn-lite';

import StationDateSpecialValue from '../../interfaces/date/StationDateSpecialValue';
import RenderMode from '../../interfaces/RenderMode';

import useSelector from '../useSelector';
import getWhenDateText from '../StationPage/getWhenDateText';
import getScheduleThreads from '../../lib/station/getScheduleThreads';

import StationNextButton from '../StationNextButton/StationNextButton';
import StationScheduleBlock from '../StationScheduleBlock/StationScheduleBlock.mobile';
import StationNoThreadsText from '../StationPage/StationTable/StationNoThreadsText';
import Spinner from '../Spinner/Spinner';

const b = B('StationBusContent');

interface IStationBusContent {
    refToScrollAfterClickNextButton?: RefObject<HTMLElement>;
}

export default memo(StationBusContent);

function StationBusContent({
    refToScrollAfterClickNextButton,
}: IStationBusContent): ReactElement {
    const scheduleBlocks = useSelector(state => state.station.scheduleBlocks);
    const whenSpecial = useSelector(state => state.station.whenSpecial);
    const whenDate = useSelector(state => state.station.whenDate);
    const tld = useSelector(state => state.tld);
    const language = useSelector(state => state.language);
    const stationId = useSelector(state => state.station.id);
    const type = useSelector(state => state.station.type);
    const event = useSelector(state => state.station.event);
    const renderMode = useSelector(state => state.environment.renderMode);
    const stop = useSelector(state => state.station.stop);

    const isAllDays = whenSpecial === StationDateSpecialValue.allDays;

    const renderLimit =
        renderMode === RenderMode.full ? scheduleBlocks.length : 10;

    const content: JSX.Element[] = [];
    let i = 0;

    // Набираем блоки расписаний, которые, с учетом фильтра, содержат хотя бы одну нитку, но
    // не более лимита, установленного для рендера
    for (; i < scheduleBlocks.length && content.length < renderLimit; i++) {
        const scheduleBlock = scheduleBlocks[i];
        const {
            title,
            number,
            transportType,
            schedule: schedules,
        } = scheduleBlock;

        const countThreads = schedules.reduce((accumulator, schedule) => {
            const {threads: allThreads} = schedule;
            const {threads, intervalThreads} = getScheduleThreads(
                allThreads,
                stop,
            );

            return accumulator + threads.length + intervalThreads.length;
        }, 0);

        if (countThreads === 0) {
            continue;
        }

        const key = `${number}-${title}-${transportType}`;

        content.push(
            <StationScheduleBlock
                key={key}
                className={b('scheduleBlock')}
                scheduleBlock={scheduleBlock}
                isAllDays={isAllDays}
                tld={tld}
                language={language}
                stationFromId={stationId}
                whenDate={whenDate || null}
                stopFilter={stop}
            />,
        );
    }

    return (
        <div className={b({isAllDays, noThreads: scheduleBlocks.length === 0})}>
            {whenDate && (
                <div className={b('whenDate')}>
                    {getWhenDateText(type, event, whenDate, whenSpecial, true)}
                </div>
            )}

            {content.length ? (
                content
            ) : (
                <StationNoThreadsText className={b('noThreadsText')} isMobile />
            )}

            {i < scheduleBlocks.length && (
                <div className={b('loadingContainer')}>
                    <Spinner size="large" />
                </div>
            )}

            {!isAllDays && (
                <StationNextButton
                    className={b('nextButton')}
                    refToScrollAfterClick={refToScrollAfterClickNextButton}
                />
            )}
        </div>
    );
}
