import React, {useCallback, useMemo} from 'react';
import B from 'bem-cn-lite';

import {momentTimezone as moment} from '../../../reexports';

import {HUMAN, ROBOT} from '../../lib/date/formats';

import DateSpecialValue from '../../interfaces/date/DateSpecialValue';
import DateRobot from '../../interfaces/date/DateRobot';
import IconGlyph from '../../interfaces/components/IconGlyph';

import getValueFromEnum from '../../lib/enum/getValueFromEnum';

import Icon from '../Icon/Icon';
import RadioGroup2 from '../RadioGroup2';

import searchFormKeyset from '../../i18n/search-form';
import segmentDatePickerKeyset from '../../i18n/segment-date-picker';

const b = B('StationDatePicker');

interface IStationDatePickerProps {
    onSpecialDateClick(special: DateSpecialValue): void;
    onDatepickerClick(): void;

    whenSpecial?: DateSpecialValue;
    whenDate?: DateRobot;
}

const DATE_PICKER = 'date-picker';

export default function StationDatePicker({
    onSpecialDateClick,
    onDatepickerClick,
    whenSpecial,
    whenDate,
}: IStationDatePickerProps): React.ReactElement {
    const buttons = useMemo(
        () => [
            {
                value: DateSpecialValue.today,
                content: searchFormKeyset('today'),
            },
            {
                value: DateSpecialValue.tomorrow,
                content: searchFormKeyset('tomorrow'),
            },
            {
                value: DateSpecialValue.allDays,
                content: searchFormKeyset('all-days'),
            },
            {
                value: DATE_PICKER,
                content: (
                    <>
                        <Icon
                            glyph={IconGlyph.calendar}
                            className={b('calendarIcon')}
                        />
                        {whenDate && !whenSpecial
                            ? moment(whenDate, ROBOT)
                                  .format(HUMAN)
                                  .toLowerCase()
                            : segmentDatePickerKeyset('button-text')}
                    </>
                ),
            },
        ],
        [whenDate, whenSpecial],
    );

    const onClick = useCallback(
        button => {
            const specialKey = getValueFromEnum(button.value, DateSpecialValue);

            if (specialKey) {
                onSpecialDateClick(specialKey);
            } else {
                onDatepickerClick();
            }
        },
        [onSpecialDateClick, onDatepickerClick],
    );

    const selectedButton = whenSpecial
        ? buttons.find(button => button.value === whenSpecial)
        : buttons.find(button => button.value === DATE_PICKER);

    return (
        <div className={b()}>
            <RadioGroup2
                buttons={buttons}
                onChange={onClick}
                selectedButton={selectedButton}
            />
        </div>
    );
}
