import React, {ReactElement, memo} from 'react';
import B from 'bem-cn-lite';
import moment from 'moment';
import upperFirst from 'lodash/upperFirst';

import {HUMAN} from '../../lib/date/formats';
import {STATION_INFORMER_UTM_MEDIUM} from '../../lib/url/constants';

import Tld from '../../interfaces/Tld';
import Lang from '../../interfaces/Lang';
import DateRobot from '../../interfaces/date/DateRobot';
import IStateFlags from '../../interfaces/state/flags/IStateFlags';
import InformerColor from '../../interfaces/components/informer/InformerColor';
import InformerTheme from '../../interfaces/components/informer/InformerTheme';
import StationDateSpecialValue from '../../interfaces/date/StationDateSpecialValue';
import IStationInformer from '../../interfaces/components/informer/IStationInformer';
import IInformerStationFromApi from '../../interfaces/components/informer/IInformerStationFromApi';

import {stationUrl} from '../../lib/url/stationUrl';
import isPlaneThreads from '../../lib/station/isPlaneThreads';
import applyInformerUtm from '../../lib/url/applyInformerUtm';
import isRailroadThreads from '../../lib/station/isRailroadThreads';
import isAllStationInformerScheduleThreads from '../../lib/informer/isAllStationInformerScheduleThreads';

import Link from '../Link';
import RaspLink from '../RaspLink/RaspLink';
import LocalTimeDisclaimer from '../LocalTimeDisclaimer/LocalTimeDisclaimer';
import StationInformerTable from '../StationInformerTable/StationInformerTable';
import StationInformerDirectionTabs from '../StationInformerDirectionTabs/StationInformerDirectionTabs';
import StationInformerPlaneTableRowDesktop from '../StationInformerPlaneTableRow/StationInformerPlaneTableRow.desktop';
import StationInformerScheduleTableRowDesktop from '../StationInformerScheduleTableRow/StationInformerScheduleTableRow.desktop';
import StationInformerRailroadTableRowDesktop from '../StationInformerRailroadTableRow/StationInformerRailroadTableRow.desktop';

import informerKeyset from '../../i18n/informers';

const b = B('StationInformerDesktop');

interface IGetTableParams {
    id: number;
    tld: Tld;
    language: Lang;
    size: number;
    stationData: IInformerStationFromApi;

    color?: InformerColor;
    theme?: InformerTheme;
    whenDate?: DateRobot;
}

function getColspanForPlaneTable(
    showTerminals: boolean,
    trusted: boolean,
): number {
    return 6 - (showTerminals ? 0 : 1) - (trusted ? 0 : 1);
}

function getTable({
    id,
    tld,
    language,
    stationData,
    size,

    color,
    theme,
    whenDate,
}: IGetTableParams): React.ReactElement | null {
    const {threads} = stationData;

    if (isAllStationInformerScheduleThreads(threads)) {
        return (
            <StationInformerTable
                rows={threads.map((thread, index) => {
                    return (
                        <StationInformerScheduleTableRowDesktop
                            id={id}
                            tld={tld}
                            language={language}
                            threads={threads}
                            index={index}
                            key={index}
                            color={color}
                            theme={theme}
                            whenDate={whenDate}
                        />
                    );
                })}
                size={size}
                color={color}
                colSpan={5}
            />
        );
    }

    if (isPlaneThreads(threads)) {
        const {companiesById, trusted, terminals, event} = stationData;

        const showTerminals = Boolean(terminals.length);

        return (
            <StationInformerTable
                rows={threads.map((thread, index) => {
                    return (
                        <StationInformerPlaneTableRowDesktop
                            tld={tld}
                            language={language}
                            threads={threads}
                            index={index}
                            companiesById={companiesById}
                            trusted={trusted}
                            showTerminals={showTerminals}
                            event={event}
                            key={index}
                            isProduction
                            color={color}
                            theme={theme}
                        />
                    );
                })}
                size={size}
                colSpan={getColspanForPlaneTable(showTerminals, trusted)}
                color={color}
            />
        );
    }

    if (isRailroadThreads(threads)) {
        return (
            <StationInformerTable
                rows={threads.map((thread, index) => {
                    return (
                        <StationInformerRailroadTableRowDesktop
                            id={id}
                            tld={tld}
                            language={language}
                            threads={threads}
                            index={index}
                            key={index}
                            color={color}
                            theme={theme}
                        />
                    );
                })}
                size={size}
                color={color}
                colSpan={3}
            />
        );
    }

    return null;
}

export default memo(StationInformerDesktop);

function StationInformerDesktop({
    tld,
    language,
    stationData,
    size,

    color,
    theme,
    informerType,
}: IStationInformer): ReactElement {
    const {
        id,
        type,
        title,
        now,
        event,

        fullTitle,
        subtype,
        whenDate,
    } = stationData;

    const stationUrlParams = {
        id,
        tld,
        language,
        flags: {} as IStateFlags,

        type,
        subtype,
    };

    const linkColor = theme === InformerTheme.black ? 'textPrimary' : undefined;
    const textDecoration =
        theme === InformerTheme.black ? 'underline' : undefined;

    return (
        <div className={b()}>
            <div className={b('header')}>
                <Link
                    className={b('titleLink')}
                    colors={linkColor}
                    href={applyInformerUtm(
                        stationUrl(stationUrlParams),
                        STATION_INFORMER_UTM_MEDIUM,
                    )}
                    target="_blank"
                >
                    <h1 className={b('h1')}>
                        <span className={b('title')}>
                            {upperFirst(fullTitle || title)}
                        </span>
                        ,&nbsp;
                        <span className={b('date')}>
                            {moment.parseZone(now).format(HUMAN)}
                        </span>
                    </h1>
                </Link>

                <LocalTimeDisclaimer />
            </div>

            <StationInformerDirectionTabs
                id={id}
                tld={tld}
                language={language}
                type={type}
                size={size}
                event={event}
                color={color}
                theme={theme}
                informerType={informerType}
            />

            {getTable({
                id,
                tld,
                language,
                stationData,
                size,

                color,
                theme,
                whenDate,
            })}

            <div className={b('footer')}>
                <div className={b('fullScheduleLinks')}>
                    <span className={b('fullSchedule')}>
                        {`${informerKeyset('full-schedule')} `}
                    </span>

                    <Link
                        href={applyInformerUtm(
                            stationUrl(stationUrlParams),
                            STATION_INFORMER_UTM_MEDIUM,
                        )}
                        target="_blank"
                        colors={linkColor}
                        className={b('link')}
                        textDecoration={textDecoration}
                    >
                        <span className={b('fullScheduleLink')}>
                            {informerKeyset('search-today')}
                        </span>
                    </Link>

                    <Link
                        href={applyInformerUtm(
                            stationUrl({
                                ...stationUrlParams,
                                special: StationDateSpecialValue.tomorrow,
                            }),
                            STATION_INFORMER_UTM_MEDIUM,
                        )}
                        target="_blank"
                        colors={linkColor}
                        className={b('link')}
                        textDecoration={textDecoration}
                    >
                        <span className={b('fullScheduleLink')}>
                            {informerKeyset('search-tomorrow')}
                        </span>
                    </Link>

                    <Link
                        href={applyInformerUtm(
                            stationUrl({
                                ...stationUrlParams,
                                special: StationDateSpecialValue.allDays,
                            }),
                            STATION_INFORMER_UTM_MEDIUM,
                        )}
                        target="_blank"
                        colors={linkColor}
                        className={b('link')}
                        textDecoration={textDecoration}
                    >
                        <span className={b('fullScheduleLink')}>
                            {informerKeyset('search-all-days')}
                        </span>
                    </Link>
                </div>

                <RaspLink
                    href={applyInformerUtm('/', STATION_INFORMER_UTM_MEDIUM)}
                    theme={theme}
                />
            </div>
        </div>
    );
}
