import React, {ReactElement} from 'react';
import B from 'bem-cn-lite';
import moment from 'moment';
import upperFirst from 'lodash/upperFirst';

import {HUMAN} from '../../lib/date/formats';
import {STATION_INFORMER_UTM_MEDIUM} from '../../lib/url/constants';

import Tld from '../../interfaces/Tld';
import Lang from '../../interfaces/Lang';
import DateRobot from '../../interfaces/date/DateRobot';
import IStateFlags from '../../interfaces/state/flags/IStateFlags';
import InformerColor from '../../interfaces/components/informer/InformerColor';
import InformerTheme from '../../interfaces/components/informer/InformerTheme';
import StationDateSpecialValue from '../../interfaces/date/StationDateSpecialValue';
import IStationInformer from '../../interfaces/components/informer/IStationInformer';
import IInformerStationFromApi from '../../interfaces/components/informer/IInformerStationFromApi';

import {stationUrl} from '../../lib/url/stationUrl';
import applyInformerUtm from '../../lib/url/applyInformerUtm';
import isPlaneThreads from '../../lib/station/isPlaneThreads';
import isRailroadThreads from '../../lib/station/isRailroadThreads';
import isAllStationInformerScheduleThreads from '../../lib/informer/isAllStationInformerScheduleThreads';

import Link from '../Link';
import RaspLink from '../RaspLink/RaspLink';
import LocalTimeDisclaimer from '../LocalTimeDisclaimer/LocalTimeDisclaimer';
import StationInformerTable from '../StationInformerTable/StationInformerTable';
import StationInformerDirectionTabs from '../StationInformerDirectionTabs/StationInformerDirectionTabs';
import StationInformerPlaneTableRowMobile from '../StationInformerPlaneTableRow/StationInformerPlaneTableRow.mobile';
import StationInformerScheduleTableRowMobile from '../StationInformerScheduleTableRow/StationInformerScheduleTableRow.mobile';
import StationInformerRailroadTableRowMobile from '../StationInformerRailroadTableRow/StationInformerRailroadTableRow.mobile';

import informerKeyset from '../../i18n/informers';

const b = B('StationInformerMobile');

interface IGetTableParams {
    id: number;
    tld: Tld;
    language: Lang;
    size: number;
    stationData: IInformerStationFromApi;

    color?: InformerColor;
    theme?: InformerTheme;
    whenDate?: DateRobot;
}

function getTable({
    id,
    tld,
    language,
    stationData,
    size,

    color,
    theme,
    whenDate,
}: IGetTableParams): React.ReactElement | null {
    const {threads} = stationData;

    if (isAllStationInformerScheduleThreads(threads)) {
        return (
            <StationInformerTable
                rows={threads.map((thread, index) => {
                    return (
                        <StationInformerScheduleTableRowMobile
                            id={id}
                            tld={tld}
                            language={language}
                            threads={threads}
                            index={index}
                            key={index}
                            color={color}
                            theme={theme}
                            whenDate={whenDate}
                        />
                    );
                })}
                size={size}
                color={color}
                colSpan={3}
                isMobile
            />
        );
    }

    if (isRailroadThreads(threads)) {
        return (
            <StationInformerTable
                rows={threads.map((thread, index) => {
                    return (
                        <StationInformerRailroadTableRowMobile
                            id={id}
                            tld={tld}
                            language={language}
                            threads={threads}
                            index={index}
                            key={index}
                            color={color}
                            theme={theme}
                        />
                    );
                })}
                size={size}
                color={color}
                colSpan={3}
                isMobile
            />
        );
    }

    if (isPlaneThreads(threads)) {
        const {companiesById, trusted, terminals, event} = stationData;

        const showTerminals = Boolean(terminals.length);

        return (
            <StationInformerTable
                rows={threads.map((thread, index) => {
                    return (
                        <StationInformerPlaneTableRowMobile
                            tld={tld}
                            language={language}
                            threads={threads}
                            index={index}
                            companiesById={companiesById}
                            trusted={trusted}
                            showTerminals={showTerminals}
                            event={event}
                            key={index}
                            isProduction
                            color={color}
                            theme={theme}
                        />
                    );
                })}
                size={size}
                colSpan={3}
                color={color}
                isMobile
            />
        );
    }

    return null;
}

export default function StationInformerMobile({
    tld,
    language,
    stationData,
    size,

    color,
    theme,
    informerType,
}: IStationInformer): ReactElement {
    const {
        id,
        type,
        title,
        now,
        event,

        fullTitle,
        subtype,
        whenDate,
    } = stationData;

    const stationUrlParams = {
        id,
        tld,
        language,
        flags: {} as IStateFlags,

        type,
        subtype,
    };

    const linkColor = theme === InformerTheme.black ? 'textPrimary' : undefined;
    const textDecoration =
        theme === InformerTheme.black ? 'underline' : undefined;

    return (
        <div className={b()}>
            <div className={b('header')}>
                <RaspLink
                    href={applyInformerUtm('/', STATION_INFORMER_UTM_MEDIUM)}
                    theme={theme}
                />

                <div className={b('titleAndDisclaimer')}>
                    <div className={b('title')}>
                        <Link
                            colors={linkColor}
                            href={applyInformerUtm(
                                stationUrl(stationUrlParams),
                                STATION_INFORMER_UTM_MEDIUM,
                            )}
                            target="_blank"
                        >
                            <h1 className={b('h1')}>
                                {`${upperFirst(fullTitle || title)}, ${moment
                                    .parseZone(now)
                                    .format(HUMAN)}`}
                            </h1>
                        </Link>
                    </div>

                    <div className={b('disclaimer')}>
                        <LocalTimeDisclaimer />
                    </div>
                </div>
            </div>

            <StationInformerDirectionTabs
                id={id}
                tld={tld}
                language={language}
                type={type}
                size={size}
                event={event}
                color={color}
                theme={theme}
                informerType={informerType}
                isMobile
            />

            {getTable({
                id,
                tld,
                language,
                stationData,
                size,

                color,
                theme,
                whenDate,
            })}

            <div className={b('footer')}>
                <Link
                    href={applyInformerUtm(
                        stationUrl(stationUrlParams),
                        STATION_INFORMER_UTM_MEDIUM,
                    )}
                    target="_blank"
                    colors={linkColor}
                    className={b('link')}
                    textDecoration={textDecoration}
                >
                    <span className={b('fullScheduleLink')}>
                        {informerKeyset('search-today')}
                    </span>
                </Link>

                <Link
                    href={applyInformerUtm(
                        stationUrl({
                            ...stationUrlParams,
                            special: StationDateSpecialValue.tomorrow,
                        }),
                        STATION_INFORMER_UTM_MEDIUM,
                    )}
                    target="_blank"
                    colors={linkColor}
                    className={b('link')}
                    textDecoration={textDecoration}
                >
                    <span className={b('fullScheduleLink')}>
                        {informerKeyset('search-tomorrow')}
                    </span>
                </Link>

                <Link
                    href={applyInformerUtm(
                        stationUrl({
                            ...stationUrlParams,
                            special: StationDateSpecialValue.allDays,
                        }),
                        STATION_INFORMER_UTM_MEDIUM,
                    )}
                    target="_blank"
                    colors={linkColor}
                    className={b('link')}
                    textDecoration={textDecoration}
                >
                    <span className={b('fullScheduleLink')}>
                        {informerKeyset('search-all-days')}
                    </span>
                </Link>
            </div>
        </div>
    );
}
