import React, {ReactElement, memo} from 'react';
import B from 'bem-cn-lite';
import moment from 'moment';

import {HUMAN_SHORT} from '../../lib/date/formats';
import {STATION_INFORMER_UTM_MEDIUM} from '../../lib/url/constants';

import Tld from '../../interfaces/Tld';
import Lang from '../../interfaces/Lang';
import IThreadPlane from '../../interfaces/state/station/IThreadPlane';
import ThreadStatus from '../../interfaces/state/station/ThreadStatus';
import InformerColor from '../../interfaces/components/informer/InformerColor';
import StationEventList from '../../interfaces/state/station/StationEventList';
import InformerTheme from '../../interfaces/components/informer/InformerTheme';
import StationCompaniesById from '../../interfaces/state/station/StationCompaniesById';
import AdditionalThreadStatus from '../../interfaces/state/station/AdditionalThreadStatus';

import getCompany from '../../lib/station/getCompany';
import needShowDate from '../StationInformer/needShowDate';
import applyInformerUtm from '../../lib/url/applyInformerUtm';
import getAviaCompanyLink from '../../lib/url/getAviaCompanyLink';
import deleteLastSymbol from '../../lib/string/deleteLastSymbol';
import getPlaneDirection from '../../lib/station/getPlaneDirection';
import getTimeFromDateMoment from '../../lib/date/getTimeFromDateMoment';
import formatCompanies from '../StationPage/StationPlaneTable/formatCompanies';
import threadTimeIsAlert from '../StationPage/StationPlaneTable/threadTimeIsAlert';
import getAviaUrlForStationThread from '../../lib/station/getAviaUrlForStationThread';
import getMessageFromStatus from '../StationPage/StationPlaneTable/getMessageFromStatus';
import threadTimeIsNotActual from '../StationPage/StationPlaneTable/threadTimeIsNotActual';
import threadTimeChangeIsAlert from '../StationPage/StationPlaneTable/threadTimeChangeIsAlert';
import getAbsoluteUrlForStaticFromBackend from '../../lib/url/getAbsoluteUrlForStaticFromBackend';

import Link from '../Link';

const b = B('StationInformerPlaneTableRowDesktop');

export default memo(StationInformerPlaneTableRowDesktop);

interface IStationInformerPlaneTableRowDesktopParams {
    tld: Tld;
    language: Lang;
    index: number;
    threads: IThreadPlane[]; // Массив может быть пустой!
    trusted: boolean; // true, если для станции доступны online статусы. Для расписания на все дни отсутствует
    companiesById: StationCompaniesById;
    showTerminals: boolean;
    event: StationEventList;
    isProduction: boolean;

    color?: InformerColor;
    theme?: InformerTheme;
}

function StationInformerPlaneTableRowDesktop({
    tld,
    language,
    index,
    threads,
    trusted,
    companiesById,
    showTerminals,
    event,
    isProduction,

    color,
    theme,
}: IStationInformerPlaneTableRowDesktopParams): ReactElement | null {
    const thread = threads[index];
    const prevThread = threads[index - 1];

    const {
        eventDt: {time, datetime},
        status,
        minutesBetweenEventDtAndActualDt,
        routeStations,
        companyId,
        aviaLink,
        number,
        codeshares,
        isSupplement,
        terminalName,
        hoursBeforeEvent,
    } = thread;

    const actialTime =
        status?.actualDt && getTimeFromDateMoment(status?.actualDt);
    const date = deleteLastSymbol(
        moment.parseZone(datetime).format(HUMAN_SHORT),
        '.',
    );
    const actualDate = status?.actualDt
        ? deleteLastSymbol(
              moment.parseZone(status?.actualDt).format(HUMAN_SHORT),
              '.',
          )
        : '';
    const isAlert = status && threadTimeIsAlert(status);
    const actualTimeIsAlert =
        status &&
        threadTimeChangeIsAlert(
            event,
            status,
            minutesBetweenEventDtAndActualDt || null,
        );
    const changeDate = needShowDate(thread, prevThread);
    const notActual =
        status &&
        status?.actualDt &&
        threadTimeIsNotActual(status, minutesBetweenEventDtAndActualDt || null);

    const direction = getPlaneDirection(routeStations, event);

    const company = getCompany(companyId, companiesById);

    const formattedCompanies = formatCompanies(
        companyId,
        number,
        companiesById,
        codeshares,
    );
    const aviaUrl = aviaLink
        ? getAviaUrlForStationThread(aviaLink, isSupplement)
        : undefined;
    const flightNumber =
        formattedCompanies.length && formattedCompanies[0].number;

    const companyTitle = company?.title;

    const actualTerminalName = status?.actualTerminalName;

    const statusMessageResult =
        status &&
        hoursBeforeEvent &&
        getMessageFromStatus(status, event, hoursBeforeEvent, undefined, true);
    const statusMessage =
        statusMessageResult && statusMessageResult.length
            ? statusMessageResult[0]
            : undefined;
    const statusMessageIsAlert =
        statusMessage &&
        (statusMessage.status === ThreadStatus.cancelled ||
            statusMessage.status === ThreadStatus.delayed ||
            statusMessage.status === AdditionalThreadStatus.changeTerminal);

    const linkColor = theme === InformerTheme.black ? 'textPrimary' : undefined;

    return (
        <tr className={b({color})}>
            <td className={b('timeColumn', {fixed: notActual && changeDate})}>
                <span className={b('scheduledTime', {isAlert, notActual})}>
                    {`${time}${changeDate ? ` ${date}` : ''}`}
                </span>

                {notActual && (
                    <span
                        className={b('actualTime', {
                            isAlert: actualTimeIsAlert,
                        })}
                    >
                        {`${actialTime}${changeDate ? ` ${actualDate}` : ''}`}
                    </span>
                )}
            </td>

            <td className={b('directionColumn')}>{direction}</td>

            <td className={b('companyLogoColumn')}>
                {company?.icon && (
                    <Link
                        href={applyInformerUtm(
                            getAviaCompanyLink({id: company.id, tld, language}),
                            STATION_INFORMER_UTM_MEDIUM,
                        )}
                    >
                        <img
                            className={b('companyLogoImg')}
                            src={getAbsoluteUrlForStaticFromBackend(
                                company.icon,
                                isProduction,
                            )}
                        />
                    </Link>
                )}
            </td>

            <td
                className={b('threadAndCompanyColumn', {
                    isLast: !showTerminals && !trusted,
                })}
            >
                {aviaUrl ? (
                    <Link
                        href={
                            aviaLink
                                ? applyInformerUtm(
                                      aviaLink,
                                      STATION_INFORMER_UTM_MEDIUM,
                                  )
                                : undefined
                        }
                        target="_blank"
                        colors={linkColor}
                    >
                        <div className={b('threadAndCompany')}>
                            <div className={b('threadNumber')}>
                                {flightNumber}
                            </div>
                            <div className={b('companyTitleLink')}>
                                {companyTitle}
                            </div>
                        </div>
                    </Link>
                ) : (
                    <div className={b('threadAndCompany')}>
                        <div className={b('threadNumber')}>{flightNumber}</div>
                        <div className={b('companyTitleLink')}>
                            {companyTitle}
                        </div>
                    </div>
                )}
            </td>

            {showTerminals && (
                <td
                    className={b('terminalColumn', {
                        isAlert:
                            actualTerminalName &&
                            actualTerminalName !== terminalName,
                        isLast: !trusted,
                    })}
                >
                    {actualTerminalName || terminalName}
                </td>
            )}

            {trusted && (
                <td
                    className={b('statusColumn', {
                        isAlert: statusMessageIsAlert,
                        notActual:
                            statusMessage &&
                            statusMessage.status === ThreadStatus.unknown,
                    })}
                >
                    <span title={statusMessage ? statusMessage.text : ''}>
                        {statusMessage ? statusMessage.text : ''}
                    </span>
                </td>
            )}
        </tr>
    );
}
