import React, {ReactElement, memo} from 'react';
import B from 'bem-cn-lite';
import moment from 'moment';

import {HUMAN_SHORT} from '../../lib/date/formats';
import {STATION_INFORMER_UTM_MEDIUM} from '../../lib/url/constants';

import Tld from '../../interfaces/Tld';
import Lang from '../../interfaces/Lang';
import IThreadPlane from '../../interfaces/state/station/IThreadPlane';
import ThreadStatus from '../../interfaces/state/station/ThreadStatus';
import InformerColor from '../../interfaces/components/informer/InformerColor';
import StationEventList from '../../interfaces/state/station/StationEventList';
import InformerTheme from '../../interfaces/components/informer/InformerTheme';
import StationCompaniesById from '../../interfaces/state/station/StationCompaniesById';

import getCompany from '../../lib/station/getCompany';
import needShowDate from '../StationInformer/needShowDate';
import applyInformerUtm from '../../lib/url/applyInformerUtm';
import getAviaCompanyLink from '../../lib/url/getAviaCompanyLink';
import deleteLastSymbol from '../../lib/string/deleteLastSymbol';
import getPlaneDirection from '../../lib/station/getPlaneDirection';
import getTimeFromDateMoment from '../../lib/date/getTimeFromDateMoment';
import formatCompanies from '../StationPage/StationPlaneTable/formatCompanies';
import threadTimeIsAlert from '../StationPage/StationPlaneTable/threadTimeIsAlert';
import getAviaUrlForStationThread from '../../lib/station/getAviaUrlForStationThread';
import getMessageFromStatus from '../StationPage/StationPlaneTable/getMessageFromStatus';
import threadTimeIsNotActual from '../StationPage/StationPlaneTable/threadTimeIsNotActual';
import threadTimeChangeIsAlert from '../StationPage/StationPlaneTable/threadTimeChangeIsAlert';
import getAbsoluteUrlForStaticFromBackend from '../../lib/url/getAbsoluteUrlForStaticFromBackend';

import Link from '../Link';

const b = B('StationInformerPlaneTableRowMobile');

export default memo(StationInformerPlaneTableRowMobile);

interface IStationInformerPlaneTableRowMobileParams {
    tld: Tld;
    language: Lang;
    index: number;
    threads: IThreadPlane[]; // Массив может быть пустой!
    trusted: boolean; // true, если для станции доступны online статусы. Для расписания на все дни отсутствует
    companiesById: StationCompaniesById;
    showTerminals: boolean;
    event: StationEventList;
    isProduction: boolean;

    color?: InformerColor;
    theme?: InformerTheme;
}

function StationInformerPlaneTableRowMobile({
    tld,
    language,
    index,
    threads,
    trusted,
    companiesById,
    showTerminals,
    event,
    isProduction,

    color,
    theme,
}: IStationInformerPlaneTableRowMobileParams): ReactElement {
    const thread = threads[index];
    const prevThread = threads[index - 1];

    const {
        companyId,
        status,
        eventDt: {datetime, time},
        minutesBetweenEventDtAndActualDt,
        routeStations,
        number,
        aviaLink,
        codeshares,
        isSupplement,
        hoursBeforeEvent,
        terminalName,
    } = thread;

    const company = getCompany(companyId, companiesById);

    const direction = getPlaneDirection(routeStations, event);

    const formattedCompanies = formatCompanies(
        companyId,
        number,
        companiesById,
        codeshares,
    );
    const aviaUrl = aviaLink
        ? getAviaUrlForStationThread(aviaLink, isSupplement)
        : undefined;
    const flightNumber =
        formattedCompanies.length && formattedCompanies[0].number;
    const companyTitle = company?.title;
    const numberAndCompanyText = `${flightNumber}${
        companyTitle ? ` ${companyTitle}` : ''
    }`;

    const statusMessageResult =
        status &&
        hoursBeforeEvent &&
        getMessageFromStatus(
            status,
            event,
            hoursBeforeEvent,
            showTerminals ? terminalName : undefined,
            true,
        );
    const statusMessage =
        statusMessageResult && statusMessageResult.length
            ? statusMessageResult[0]
            : undefined;
    const statusMessageText = statusMessage ? statusMessage.text : '';

    const actialTime =
        status?.actualDt && getTimeFromDateMoment(status?.actualDt);
    const date = deleteLastSymbol(
        moment.parseZone(datetime).format(HUMAN_SHORT),
        '.',
    );
    const actualDate = status?.actualDt
        ? deleteLastSymbol(
              moment.parseZone(status?.actualDt).format(HUMAN_SHORT),
              '.',
          )
        : '';
    const isAlert = status && threadTimeIsAlert(status);
    const actualTimeIsAlert =
        status &&
        threadTimeChangeIsAlert(
            event,
            status,
            minutesBetweenEventDtAndActualDt || null,
        );
    const changeDate = needShowDate(thread, prevThread);
    const notActual =
        status &&
        status?.actualDt &&
        threadTimeIsNotActual(status, minutesBetweenEventDtAndActualDt || null);

    const linkColor = theme === InformerTheme.black ? 'textPrimary' : undefined;

    return (
        <tr className={b({color})}>
            <td className={b('companyLogoColumn', {color})}>
                {company?.icon && (
                    <Link
                        href={getAviaCompanyLink({
                            id: company.id,
                            tld,
                            language,
                        })}
                    >
                        <img
                            className={b('companyLogoImg')}
                            src={getAbsoluteUrlForStaticFromBackend(
                                company.icon,
                                isProduction,
                            )}
                        />
                    </Link>
                )}
            </td>

            <td className={b('threadInfo')}>
                <div className={b('direction')}>{direction}</div>

                <div className={b('numberAndCompany')}>
                    {aviaUrl ? (
                        <Link
                            href={
                                aviaLink
                                    ? applyInformerUtm(
                                          aviaLink,
                                          STATION_INFORMER_UTM_MEDIUM,
                                      )
                                    : undefined
                            }
                            target="_blank"
                            colors={linkColor}
                        >
                            <div className={b('numberAndCompanyLink')}>
                                {numberAndCompanyText}
                            </div>
                        </Link>
                    ) : (
                        <div className={b('numberAndCompanyText')}>
                            {numberAndCompanyText}
                        </div>
                    )}
                </div>

                {trusted && (
                    <div
                        className={b('status', {
                            isAlert:
                                statusMessage &&
                                statusMessage.status === ThreadStatus.cancelled,
                            isSecondaryInfo:
                                (statusMessage &&
                                    statusMessage.status ===
                                        ThreadStatus.unknown) ||
                                theme === InformerTheme.black,
                        })}
                        title={statusMessageText}
                    >
                        {statusMessageText}
                    </div>
                )}
            </td>

            <td className={b('timeColumn', {color})}>
                <div className={b('scheduledTime', {isAlert, notActual})}>
                    {`${changeDate ? `${date}` : ''} ${time}`}
                </div>

                {notActual && (
                    <div
                        className={b('actualTime', {
                            isAlert: actualTimeIsAlert,
                        })}
                    >
                        {`${changeDate ? `${actualDate}` : ''} ${actialTime}`}
                    </div>
                )}
            </td>
        </tr>
    );
}
