import React, {ReactElement, memo} from 'react';
import B from 'bem-cn-lite';
import moment from 'moment';
import upperFirst from 'lodash/upperFirst';

import {HUMAN_SHORT} from '../../lib/date/formats';
import {STATION_INFORMER_UTM_MEDIUM} from '../../lib/url/constants';

import Tld from '../../interfaces/Tld';
import Lang from '../../interfaces/Lang';
import IThreadRailroad from '../../interfaces/state/station/IThreadRailroad';
import InformerColor from '../../interfaces/components/informer/InformerColor';
import InformerTheme from '../../interfaces/components/informer/InformerTheme';

import getThreadUrl from '../../lib/url/getThreadUrl';
import getSubtitle from '../../lib/station/getSubtitle';
import needShowDate from '../StationInformer/needShowDate';
import applyInformerUtm from '../../lib/url/applyInformerUtm';
import deleteLastSymbol from '../../lib/string/deleteLastSymbol';
import getTimeFromDateMoment from '../../lib/date/getTimeFromDateMoment';
import getStaticTransportTypeName from '../../lib/getStaticTransportTypeName';

import TransportIcon from '../TransportIcon/TransportIcon';
import InformerDirectionTitle from '../InformerDirectionTitle/InformerDirectionTitle';

const b = B('StationInformerRailroadTableRowDesktop');

export default memo(StationInformerRailroadTableRowDesktop);

interface IStationInformerRailroadTableRowDesktopParams {
    id: number;
    tld: Tld;
    language: Lang;
    threads: IThreadRailroad[];
    index: number;

    color?: InformerColor;
    theme?: InformerTheme;
}

function StationInformerRailroadTableRowDesktop({
    id,
    tld,
    language,
    threads,
    index,

    color,
    theme,
}: IStationInformerRailroadTableRowDesktopParams): ReactElement {
    const thread = threads[index];

    const {
        departureFrom,
        isExpress,
        isAeroExpress,
        title,
        number,
        transportType,
        deluxeTrainTitle,
        transportSubtype,
        canonicalUid,
        departure,
        eventDt: {datetime},
    } = thread;

    const departureFromMoment = moment.parseZone(departureFrom);
    const showDate = needShowDate(thread, threads[index - 1]);
    const subtitle = getSubtitle(
        transportType,
        isAeroExpress,
        transportSubtype,
        deluxeTrainTitle,
    );

    const threadUrl = applyInformerUtm(
        getThreadUrl({
            tld,
            language,
            canonicalUid,
            departure,
            stationFromId: departureFrom ? id : undefined,
            departureFromMoment,
        }),
        STATION_INFORMER_UTM_MEDIUM,
    );

    return (
        <tr className={b({color, theme})}>
            <td className={b('timeColumn')}>
                {departureFrom
                    ? getTimeFromDateMoment(departureFrom)
                    : datetime
                    ? getTimeFromDateMoment(datetime)
                    : ''}
                {showDate &&
                    deleteLastSymbol(
                        ` ${
                            departureFrom
                                ? departureFromMoment.format(HUMAN_SHORT)
                                : moment.parseZone(datetime).format(HUMAN_SHORT)
                        }`,
                        '.',
                    )}
            </td>

            <td className={b('transportTypeColumn')}>
                <TransportIcon
                    className={b('transportTypeIcon')}
                    transportType={transportType}
                    isExpress={isExpress}
                    isAeroExpress={isAeroExpress}
                    colored
                />
            </td>

            <td className={b('directionColumn')}>
                <div className={b('direction')}>
                    <InformerDirectionTitle
                        title={title}
                        href={threadUrl}
                        className={b('threadTitle')}
                        theme={theme}
                    />

                    {` ${number}`}

                    {` ${upperFirst(
                        getStaticTransportTypeName(transportType),
                    )}`}

                    {subtitle && (
                        <span className={b('subtitle')}>{` ${subtitle}`}</span>
                    )}
                </div>
            </td>
        </tr>
    );
}
