import React, {ReactElement, memo} from 'react';
import B from 'bem-cn-lite';
import moment from 'moment';
import upperFirst from 'lodash/upperFirst';

import {HUMAN_SHORT} from '../../lib/date/formats';
import {STATION_INFORMER_UTM_MEDIUM} from '../../lib/url/constants';

import Tld from '../../interfaces/Tld';
import Lang from '../../interfaces/Lang';
import IThreadRailroad from '../../interfaces/state/station/IThreadRailroad';
import InformerColor from '../../interfaces/components/informer/InformerColor';
import InformerTheme from '../../interfaces/components/informer/InformerTheme';

import getThreadUrl from '../../lib/url/getThreadUrl';
import getSubtitle from '../../lib/station/getSubtitle';
import needShowDate from '../StationInformer/needShowDate';
import applyInformerUtm from '../../lib/url/applyInformerUtm';
import deleteLastSymbol from '../../lib/string/deleteLastSymbol';
import getTimeFromDateMoment from '../../lib/date/getTimeFromDateMoment';
import getStaticTransportTypeName from '../../lib/getStaticTransportTypeName';
import needChangeDateForRailroadThread from './needChangeDateForRailroadThread';

import Link from '../Link';
import TransportIcon from '../TransportIcon/TransportIcon';

const b = B('StationInformerRailroadTableRowMobile');

export default memo(StationInformerRailroadTableRowMobile);

interface IStationInformerRailroadTableRowMobileParams {
    id: number;
    tld: Tld;
    language: Lang;
    threads: IThreadRailroad[];
    index: number;

    color?: InformerColor;
    theme?: InformerTheme;
}

function StationInformerRailroadTableRowMobile({
    id,
    tld,
    language,
    threads,
    index,

    color,
    theme,
}: IStationInformerRailroadTableRowMobileParams): ReactElement {
    const thread = threads[index];

    const {
        transportType,
        isExpress,
        isAeroExpress,
        title,
        canonicalUid,
        departure,
        departureFrom,
        number,
        transportSubtype,
        deluxeTrainTitle,
        eventDt: {time, datetime},
    } = thread;

    const departureFromMoment = moment.parseZone(departureFrom);
    const threadUrl = applyInformerUtm(
        getThreadUrl({
            tld,
            language,
            canonicalUid,
            departure,
            stationFromId: departureFrom ? id : undefined,
            departureFromMoment,
        }),
        STATION_INFORMER_UTM_MEDIUM,
    );

    const subtitle = getSubtitle(
        transportType,
        isAeroExpress,
        transportSubtype,
        deluxeTrainTitle,
    );

    const datetimeMoment = moment.parseZone(datetime);
    const needChangeTime = needChangeDateForRailroadThread(thread);
    const showDate = needShowDate(thread, threads[index - 1]);

    const formatedTime =
        (departureFrom && getTimeFromDateMoment(departureFrom)) || time;

    const departureFromText = `${deleteLastSymbol(
        departureFromMoment.format(HUMAN_SHORT),
        '.',
    )} ${formatedTime}`;
    const dateMomentText = `${deleteLastSymbol(
        datetimeMoment.format(HUMAN_SHORT),
        '.',
    )} ${datetime && getTimeFromDateMoment(datetime)}`;

    const linkColor = theme === InformerTheme.black ? 'textPrimary' : undefined;

    return (
        <tr className={b({color, theme})}>
            <td className={b('transportTypeColumn')}>
                <TransportIcon
                    className={b('transportTypeIcon')}
                    transportType={transportType}
                    isExpress={isExpress}
                    isAeroExpress={isAeroExpress}
                    colored
                />
            </td>

            <td className={b('threadTitle')}>
                <div className={b('titleLink')}>
                    <Link href={threadUrl} target="_blank" colors={linkColor}>
                        <span className={b('title')}>{title}</span>
                    </Link>

                    <span
                        className={b('gradient', {
                            changeColor: index % 2 === 0,
                        })}
                    />
                </div>

                <span className={b('info')}>
                    {number}

                    {` ${upperFirst(
                        getStaticTransportTypeName(transportType),
                    )}`}

                    {subtitle && (
                        <span className={b('subtitle')}>{` ${subtitle}`}</span>
                    )}
                </span>
            </td>

            <td className={b('timeColumn')}>
                {needChangeTime ? (
                    <div className={b('changeTime')}>
                        <span className={b('outdatedTime')}>
                            {departureFromText}
                        </span>

                        {dateMomentText}
                    </div>
                ) : showDate ? (
                    dateMomentText
                ) : (
                    formatedTime
                )}
            </td>
        </tr>
    );
}
