import React, {ReactElement, memo} from 'react';
import B from 'bem-cn-lite';
import moment from 'moment';
import upperFirst from 'lodash/upperFirst';

import {HUMAN_SHORT} from '../../lib/date/formats';
import {STATION_INFORMER_UTM_MEDIUM} from '../../lib/url/constants';

import Tld from '../../interfaces/Tld';
import Lang from '../../interfaces/Lang';
import DateRobot from '../../interfaces/date/DateRobot';
import InformerColor from '../../interfaces/components/informer/InformerColor';
import InformerTheme from '../../interfaces/components/informer/InformerTheme';
import AllStationInformerScheduleThreadType from '../../interfaces/components/informer/AllStationInformerScheduleThreadType';

import getThreadUrl from '../../lib/url/getThreadUrl';
import needShowDate from '../StationInformer/needShowDate';
import applyInformerUtm from '../../lib/url/applyInformerUtm';
import deleteLastSymbol from '../../lib/string/deleteLastSymbol';
import getTimeFromDateMoment from '../../lib/date/getTimeFromDateMoment';
import isStationInformerScheduleThread from '../../lib/informer/isStationInformerScheduleThread';
import isStationInformerScheduleIntervalThread from '../../lib/informer/isStationInformerScheduleIntervalThread';

import Link from '../Link';
import TransportIcon from '../TransportIcon/TransportIcon';

const b = B('StationInformerScheduleTableRowDesktop');

interface IStationInformerScheduleTableRowDesktopParams {
    id: number;
    tld: Tld;
    language: Lang;
    threads: AllStationInformerScheduleThreadType[];
    index: number; // индекс текущего элемента в таблице

    color?: InformerColor;
    theme?: InformerTheme;
    whenDate?: DateRobot;
}

export default memo(StationInformerScheduleTableRowDesktop);

function StationInformerScheduleTableRowDesktop({
    id,
    tld,
    language,
    threads,
    index,

    color,
    theme,
    whenDate,
}: IStationInformerScheduleTableRowDesktopParams): ReactElement {
    const thread = threads[index];

    const {transportType, number, title, canonicalUid} = thread;

    const url = getThreadUrl({
        canonicalUid,
        tld,
        language,
        stationFromId: id,
        departureFromDateRobot: whenDate ?? undefined,
    });

    const showDate = needShowDate(thread, threads[index - 1]);
    const linkColor = theme === InformerTheme.black ? 'textPrimary' : undefined;

    return (
        <tr className={b({color, theme})}>
            <td className={b('timeColumn')}>
                {isStationInformerScheduleThread(thread)
                    ? getTimeFromDateMoment(thread.departureFrom)
                    : ''}
            </td>

            <td className={b('dateColumn')}>
                {showDate
                    ? isStationInformerScheduleThread(thread)
                        ? deleteLastSymbol(
                              moment
                                  .parseZone(thread.departureFrom)
                                  .format(HUMAN_SHORT),
                              '.',
                          )
                        : deleteLastSymbol(
                              moment.parseZone(thread.date).format(HUMAN_SHORT),
                              '.',
                          )
                    : ''}
            </td>

            <td className={b('transportTypeColumn')}>
                <TransportIcon
                    className={b('transportTypeIcon')}
                    transportType={transportType}
                    colored
                />
            </td>

            <td
                className={b('numberColumn', {
                    large:
                        threads.some(t => t.number.length > 4) && index === 0,
                })}
            >
                <Link
                    href={applyInformerUtm(url, STATION_INFORMER_UTM_MEDIUM)}
                    target="_blank"
                    colors={linkColor}
                >
                    {deleteLastSymbol(number, '/')}
                </Link>
            </td>

            <td className={b('threadInfoColumn')}>
                <div className={b('threadInfo')}>
                    <Link
                        href={applyInformerUtm(
                            url,
                            STATION_INFORMER_UTM_MEDIUM,
                        )}
                        target="_blank"
                        colors={linkColor}
                        className={b('titleLink')}
                    >
                        {title}
                    </Link>

                    {isStationInformerScheduleIntervalThread(thread) && (
                        <>
                            <span className={b('density')}>
                                {upperFirst(thread.density)}
                            </span>

                            <span className={b('comment')}>
                                {thread.comment}
                            </span>
                        </>
                    )}
                </div>
            </td>
        </tr>
    );
}
