import React, {ReactElement, memo} from 'react';
import B from 'bem-cn-lite';
import {upperFirst} from 'lodash';

import {STATION_INFORMER_UTM_MEDIUM} from '../../lib/url/constants';

import Tld from '../../interfaces/Tld';
import Lang from '../../interfaces/Lang';
import DateRobot from '../../interfaces/date/DateRobot';
import InformerColor from '../../interfaces/components/informer/InformerColor';
import InformerTheme from '../../interfaces/components/informer/InformerTheme';
import AllStationInformerScheduleThreadType from '../../interfaces/components/informer/AllStationInformerScheduleThreadType';

import getThreadUrl from '../../lib/url/getThreadUrl';
import applyInformerUtm from '../../lib/url/applyInformerUtm';
import getTimeFromDateMoment from '../../lib/date/getTimeFromDateMoment';
import isStationInformerScheduleThread from '../../lib/informer/isStationInformerScheduleThread';
import isStationInformerScheduleIntervalThread from '../../lib/informer/isStationInformerScheduleIntervalThread';

import Link from '../Link';
import TransportIcon from '../TransportIcon/TransportIcon';

const b = B('StationInformerScheduleTableRowMobile');

export default memo(StationInformerScheduleTableRowMobile);

interface IStationInformerScheduleTableRowMobileParams {
    id: number;
    tld: Tld;
    language: Lang;
    threads: AllStationInformerScheduleThreadType[];
    index: number;

    color?: InformerColor;
    theme?: InformerTheme;
    whenDate?: DateRobot;
}

function StationInformerScheduleTableRowMobile({
    id,
    tld,
    language,
    threads,
    index,

    color,
    theme,
    whenDate,
}: IStationInformerScheduleTableRowMobileParams): ReactElement {
    const thread = threads[index];

    const {transportType, number, title, canonicalUid} = thread;

    const url = getThreadUrl({
        canonicalUid,
        tld,
        language,
        stationFromId: id,
        departureFromDateRobot: whenDate ?? undefined,
    });

    const linkColor = theme === InformerTheme.black ? 'textPrimary' : undefined;

    return (
        <tr className={b({color, theme})}>
            <td className={b('transportTypeColumn')}>
                <TransportIcon
                    className={b('transportTypeIcon')}
                    transportType={transportType}
                    colored
                />
            </td>

            <td className={b('threadInfoColumn')}>
                <div className={b('threadInfo')}>
                    <Link
                        href={applyInformerUtm(
                            url,
                            STATION_INFORMER_UTM_MEDIUM,
                        )}
                        target="_blank"
                        colors={linkColor}
                    >
                        <span className={b('title')}>
                            {number && (
                                <span className={b('number')}>{number}</span>
                            )}

                            {title}
                        </span>
                    </Link>

                    <span
                        className={b('gradient', {
                            changeColor: index % 2 === 0,
                        })}
                    />
                </div>

                {isStationInformerScheduleIntervalThread(thread) && (
                    <div className={b('intervalThreadInfo')}>
                        <span className={b('density')}>
                            {upperFirst(thread.density)}
                        </span>

                        <span className={b('comment')}>{thread.comment}</span>
                    </div>
                )}
            </td>

            <td className={b('timeColumn')}>
                {isStationInformerScheduleThread(thread)
                    ? getTimeFromDateMoment(thread.departureFrom)
                    : ''}
            </td>
        </tr>
    );
}
