import React, {
    ReactElement,
    memo,
    useCallback,
    useMemo,
    RefObject,
} from 'react';
import B from 'bem-cn-lite';
import moment from 'moment';

import {ROBOT, HUMAN} from '../../lib/date/formats';

import StationDateSpecialValue from '../../interfaces/date/StationDateSpecialValue';

import scrollWindow from '../../../client/lib/scrollWindow';
import {stationUrl} from '../../lib/url/stationUrl';
import isDateRobot from '../../lib/date/isDateRobot';
import {useFlags} from '../../hooks/useFlags';
import useSelector from '../useSelector';

import Link from '../Link';

import stationKeyset from '../../i18n/station';

const b = B('StationNextButton');

interface IStationNextButton {
    className?: string;
    refToScrollAfterClick?: RefObject<HTMLElement>;
}

export default memo(StationNextButton);

function StationNextButton({
    className,
    refToScrollAfterClick,
}: IStationNextButton): ReactElement | null {
    const id = useSelector(state => state.station.id);
    const event = useSelector(state => state.station.event);
    const whenDate = useSelector(state => state.station.whenDate);
    const whenSpecial = useSelector(state => state.station.whenSpecial);
    const type = useSelector(state => state.station.type);
    const subtype = useSelector(state => state.station.currentSubtype);
    const mainSubtype = useSelector(state => state.station.mainSubtype);
    const directionCode = useSelector(state => state.station.directionCode);
    const tld = useSelector(state => state.tld);
    const language = useSelector(state => state.language);
    const flags = useFlags();

    const whenMoment = whenDate ? moment(whenDate) : null;

    const nextButton = useMemo(() => {
        const commonParams = {
            id,
            tld,
            language,
            flags,
            type,
            subtype: subtype ?? undefined,
            mainSubtype: mainSubtype ?? undefined,
            isMobile: true,
            event,
            direction: directionCode ?? undefined,
        };

        switch (whenSpecial) {
            case StationDateSpecialValue.today:
                return {
                    text: stationKeyset('schedule-on-tomorrow'),
                    link: stationUrl({
                        ...commonParams,
                        special: StationDateSpecialValue.tomorrow,
                    }),
                };
        }

        if (whenMoment) {
            const newDateMoment = whenMoment.clone().add(1, 'day');
            const newDateRobot = newDateMoment.format(ROBOT);

            if (!isDateRobot(newDateRobot)) {
                return;
            }

            return {
                text: stationKeyset('schedule-on-date', {
                    date: newDateMoment.format(HUMAN),
                }),
                link: stationUrl({
                    ...commonParams,
                    date: newDateRobot,
                }),
            };
        }
    }, [
        directionCode,
        event,
        flags,
        id,
        language,
        mainSubtype,
        subtype,
        tld,
        type,
        whenMoment,
        whenSpecial,
    ]);

    const onClickNextButton = useCallback(() => {
        if (nextButton?.link) {
            scrollWindow(refToScrollAfterClick || 0);
        }
    }, [nextButton, refToScrollAfterClick]);

    if (!nextButton) {
        return null;
    }

    return (
        <div className={b(undefined, className)}>
            <Link
                className={b('link')}
                href={nextButton.link}
                onClick={onClickNextButton}
            >
                {nextButton.text}
            </Link>
        </div>
    );
}
