import React, {useState, useMemo, useCallback} from 'react';
import B from 'bem-cn-lite';

import StationType from '../../../interfaces/state/station/StationType';
import ICityData from '../../../interfaces/state/station/ICityData';
import StationSubtype from '../../../interfaces/state/station/StationSubtype';

import useSelector from '../../useSelector';
import {stationUrl} from '../../../lib/url/stationUrl';
import getValueFromEnum from '../../../lib/enum/getValueFromEnum';

import Link from '../../Link';
import Arrow from '../../../components/Arrow/Arrow';

import stationKeyset from '../../../i18n/station';

const b = B('CityStations');

const STATIONS_ORDER = [
    StationType.plane,
    StationType.railroad,
    StationType.bus,
    StationType.water,
];

const COUNT_FOR_PREVIEW = 3;
const COUNT_FOR_SHOW_BUTTON = 6;

interface ICityStationsProps {
    type: StationType;
    cityData: ICityData;

    currentSubtype?: StationSubtype;
    className?: string;
    onLinkClick?: () => void;
}

function CityStations({
    type,
    currentSubtype,
    cityData,
    className,
    onLinkClick,
}: ICityStationsProps): React.ReactElement | null {
    const tld = useSelector(state => state.tld);
    const language = useSelector(state => state.language);
    const [showAll, setShowAll] = useState(false);
    const stationId = useSelector(state => state.station.id);

    const {settlement, cityStations} = cityData;

    const onShowAllClick = useCallback(() => setShowAll(true), [setShowAll]);

    const stations = useMemo(
        () =>
            Object.entries(cityStations)
                .sort((aStation, bStation) => {
                    const indexA = STATIONS_ORDER.findIndex(
                        typeStation => typeStation === aStation[0],
                    );
                    const indexB = STATIONS_ORDER.findIndex(
                        typeStation => typeStation === bStation[0],
                    );

                    if (indexA === -1 && indexB === -1) {
                        return 0;
                    }

                    if (indexA === -1) {
                        return 1;
                    }

                    if (indexB === -1) {
                        return -1;
                    }

                    return indexA - indexB;
                })
                .map(([typeOfStations, stationsList]) => {
                    const typeOfStationsAsType = getValueFromEnum(
                        typeOfStations,
                        StationType,
                    );
                    const isBus = typeOfStations === StationType.bus;
                    const needShowButton =
                        isBus &&
                        !showAll &&
                        stationsList &&
                        stationsList.length > COUNT_FOR_SHOW_BUTTON;

                    return (
                        <div className={b('type')} key={typeOfStations}>
                            <div className={b('typeTitle')}>
                                {stationKeyset(`type-${typeOfStations}`)}
                            </div>
                            <div className={b('typeList')}>
                                {stationsList &&
                                    stationsList.map((station, index) => (
                                        <div
                                            className={b('cityStation', {
                                                dontShow:
                                                    needShowButton &&
                                                    index >= COUNT_FOR_PREVIEW,
                                            })}
                                            key={station.id}
                                        >
                                            {station.id !== stationId ? (
                                                <Link
                                                    className={b(
                                                        'cityStationLink',
                                                    )}
                                                    href={stationUrl({
                                                        id: station.id,
                                                        tld,
                                                        language,
                                                        type: typeOfStationsAsType,
                                                        subtypes:
                                                            station.subtypes,
                                                        subtype:
                                                            type ===
                                                            typeOfStationsAsType
                                                                ? currentSubtype
                                                                : undefined,
                                                        mainSubtype:
                                                            station.mainSubtype,
                                                    })}
                                                    onClick={onLinkClick}
                                                >
                                                    {station.title}
                                                </Link>
                                            ) : (
                                                station.title
                                            )}
                                        </div>
                                    ))}
                                {needShowButton && (
                                    <div
                                        className={b('showAllBusStations')}
                                        onClick={onShowAllClick}
                                    >
                                        {stationKeyset('show-all-bus-stations')}
                                        <Arrow
                                            className={b('showAllIcon')}
                                            direction="down"
                                            size="m"
                                        />
                                    </div>
                                )}
                            </div>
                        </div>
                    );
                }),
        [
            cityStations,
            showAll,
            onShowAllClick,
            tld,
            language,
            type,
            currentSubtype,
            onLinkClick,
            stationId,
        ],
    );

    if (!settlement) {
        return null;
    }

    const {title, titleGenitive} = settlement;

    if (stations.length === 0) {
        return null;
    }

    return (
        <div className={b(undefined, className)}>
            <div className={b('title')}>
                {stationKeyset('city-stations', {
                    cityTitleGenitive: titleGenitive ?? title,
                })}
            </div>
            <div className={b('stations')}>{stations}</div>
        </div>
    );
}

export default React.memo(CityStations);
