import React from 'react';
import B from 'bem-cn-lite';

import Tld from '../../../interfaces/Tld';
import Lang from '../../../interfaces/Lang';
import StationType from '../../../interfaces/state/station/StationType';
import ICityData from '../../../interfaces/state/station/ICityData';
import CityStationsType from '../../../interfaces/state/station/CityStations';
import StationSubtype from '../../../interfaces/state/station/StationSubtype';

import useSelector from '../../useSelector';

import {stationUrl} from '../../../lib/url/stationUrl';
import getValueFromEnum from '../../../lib/enum/getValueFromEnum';

import Link from '../../Link';

import stationKeyset from '../../../i18n/station';

const b = B('CityStations');

const STATIONS_ORDER = [
    StationType.plane,
    StationType.railroad,
    StationType.bus,
    StationType.water,
];

function getStations(
    stations: CityStationsType,
    type: StationType,
    tld: Tld,
    language: Lang,

    currentSubtype?: StationSubtype,
    onLinkClick?: () => void,
): React.ReactElement[] {
    return Object.entries(stations)
        .sort((aStation, bStation) => {
            const indexA = STATIONS_ORDER.findIndex(
                typeStation => typeStation === aStation[0],
            );
            const indexB = STATIONS_ORDER.findIndex(
                typeStation => typeStation === bStation[0],
            );

            if (indexA === -1 && indexB === -1) {
                return 0;
            }

            if (indexA === -1) {
                return 1;
            }

            if (indexB === -1) {
                return -1;
            }

            return indexA - indexB;
        })
        .map(([typeOfStations, stationsList]) => {
            const typeOfStationsAsType = getValueFromEnum(
                typeOfStations,
                StationType,
            );

            return (
                <div className={b('type')} key={typeOfStations}>
                    <div className={b('typeTitle')}>
                        {stationKeyset(`type-${typeOfStations}`)}
                    </div>
                    <div className={b('typeList')}>
                        {stationsList &&
                            stationsList.map(station => (
                                <div
                                    className={b('cityStation')}
                                    key={station.id}
                                >
                                    <Link
                                        className={b('cityStationLink')}
                                        href={stationUrl({
                                            id: station.id,
                                            tld,
                                            language,
                                            type: typeOfStationsAsType,
                                            subtypes: station.subtypes,
                                            subtype:
                                                type === typeOfStationsAsType
                                                    ? currentSubtype
                                                    : undefined,
                                            mainSubtype: station.mainSubtype,
                                        })}
                                        onClick={onLinkClick}
                                    >
                                        {station.title}
                                    </Link>
                                </div>
                            ))}
                    </div>
                </div>
            );
        });
}

interface ICityStationsProps {
    type: StationType;

    currentSubtype?: StationSubtype;
    cityData?: ICityData;
    className?: string;
    onLinkClick?: () => void;
}

function CityStations({
    type,
    currentSubtype,
    cityData,
    className,
    onLinkClick,
}: ICityStationsProps): React.ReactElement | null {
    const tld = useSelector(state => state.tld);
    const language = useSelector(state => state.language);

    if (!cityData || !cityData.settlement) {
        return null;
    }

    const {settlement, cityStations} = cityData;
    const {title, titleGenitive} = settlement;

    const stations = getStations(
        cityStations,
        type,
        tld,
        language,
        currentSubtype,
        onLinkClick,
    );

    if (stations.length === 0) {
        return null;
    }

    return (
        <div className={b(undefined, className)}>
            <div className={b('title')}>
                {stationKeyset('city-stations', {
                    cityTitleGenitive: titleGenitive ?? title,
                })}
            </div>

            <div className={b('stations')}>{stations}</div>
        </div>
    );
}

export default React.memo(CityStations);
