import React, {
    useEffect,
    useState,
    useCallback,
    useMemo,
    useRef,
    MutableRefObject,
} from 'react';
import B from 'bem-cn-lite';
import upperFirst from 'lodash/upperFirst';
import isEqual from 'lodash/isEqual';
import throttle from 'lodash/throttle';

import {DIRECTION_CODE_ALL} from '../../../lib/station/stationConstants';

import StationEventList from '../../../interfaces/state/station/StationEventList';
import StationSubtype from '../../../interfaces/state/station/StationSubtype';
import IDirection from '../../../interfaces/state/station/IDirection';

import {stationUrl} from '../../../lib/url/stationUrl';
import {reachGoal} from '../../../lib/yaMetrika';
import {makeCacheable} from '../../../lib/cache';
import {useFlags} from '../../../hooks/useFlags';
import useSelector from '../../useSelector';
import useDispatch from '../../useDispatch';

import {setMoreDirectionsIsOpened} from '../../../actions/station';

import Link from '../../Link';
import Arrow from '../../Arrow/Arrow';
import Popup from '../../Popup/Popup';

import stationKeyset from '../../../i18n/station';

const b = B('DirectionTabs');

const onSuburbanArrivalClick = (): void => {
    reachGoal('station_suburban_arrival_click');
};

const getOnDirectionLinkClick = makeCacheable(
    (direction: IDirection) => (): void => {
        reachGoal(
            `station_direction_${
                direction.code === DIRECTION_CODE_ALL
                    ? DIRECTION_CODE_ALL
                    : 'specific'
            }_click`,
        );
    },
);

type DirectionWithWidth = IDirection & {width: number};
type DirectionsWithWidth = {
    forDirections?: IDirection[]; // Массив направлений, для элементов которого вычисляется ширина
    directionsWithWidth?: DirectionWithWidth[];
};
type DirectionWithTotalWidth = DirectionWithWidth & {totalWidth: number};

interface IDirectionTabsProps {
    className?: string;
}

export default React.memo(DirectionTabs);

function DirectionTabs({className}: IDirectionTabsProps): React.ReactElement {
    const dispatch = useDispatch();

    const tld = useSelector(state => state.tld);
    const language = useSelector(state => state.language);
    const flags = useFlags();
    const event = useSelector(state => state.station.event);
    const id = useSelector(state => state.station.id);
    const type = useSelector(state => state.station.type);
    const terminalName = useSelector(state => state.station.terminalName);
    const moreDirectionsIsOpened = useSelector(
        state => state.station.moreDirectionsIsOpened,
    );
    const directions = useSelector(state => state.station.directions);
    const directionCode = useSelector(state => state.station.directionCode);
    const whenDate = useSelector(state => state.station.whenDate);
    const whenSpecial = useSelector(state => state.station.whenSpecial);
    const currentSubtype = useSelector(state => state.station.currentSubtype);
    const mainSubtype = useSelector(state => state.station.mainSubtype);

    const [directionsWithWidth, setDirectionsWithWidth] =
        useState<DirectionsWithWidth>({}); // directions с информацией о ширине каждого
    const [widthDirectionsContainer, setWidthDirectionsContainer] = useState<
        number | undefined
    >(undefined); // Ширина блока направлений и кнопки "еще"
    const [widthMoreDirections, setWidthMoreDirections] = useState<
        number | undefined
    >(undefined); // Ширина кнопки "еще"
    const [moreDirections, setMoreDirections] = useState<IDirection[]>([]); // Направления, который попадут в попап кнопки "еще"
    const [sortedDirections, setSortedDirections] = useState<
        IDirection[] | undefined
    >(directions); // Отсортированный список направлений

    const refDirectionsContainer = useRef<HTMLDivElement | null>(null);
    const refFakeDirections = useRef<HTMLDivElement | null>(null);
    const refMoreDirectionsContainer = useRef<HTMLDivElement | null>(null);

    const isSuburbanWithDirections =
        currentSubtype === StationSubtype.suburban && Boolean(directions);

    const commonUrlParameters = useMemo(
        () => ({
            id,
            subtype: currentSubtype,
            mainSubtype,
            tld,
            language,
            flags,
            date: whenDate,
            special: whenSpecial,
            type,
            terminalName,
        }),
        [
            currentSubtype,
            flags,
            id,
            language,
            mainSubtype,
            terminalName,
            tld,
            type,
            whenDate,
            whenSpecial,
        ],
    );

    useEffect(() => {
        if (!refFakeDirections.current) {
            return;
        }

        if (directionsWithWidth?.forDirections === directions) {
            // Если уже вычисляли ширину для заданных направлений, то выходим
            return;
        }

        const result: DirectionsWithWidth & {
            directionsWithWidth: IDirection[];
        } = {
            forDirections: directions,
            directionsWithWidth: [],
        };

        if (directions) {
            const {childNodes} = refFakeDirections.current;

            for (let i = 0; i < childNodes.length; i++) {
                const element = childNodes[i];

                if (element instanceof HTMLDivElement) {
                    result.directionsWithWidth.push({
                        ...directions[i],
                        width: element.offsetWidth + 1, // Компенсируем округление offsetWidth браузера до целых
                    });
                }
            }
        }

        setDirectionsWithWidth(result);
    }, [directions, directionsWithWidth]);

    const calculateWidthContainerDirections = useCallback(() => {
        // Функция по вычислению ширины контейнера для направлений и кнопки "еще"
        if (!refDirectionsContainer.current) {
            return;
        }

        const width = refDirectionsContainer.current.clientWidth;

        if (width !== widthDirectionsContainer) {
            setWidthDirectionsContainer(width);
        }
    }, [widthDirectionsContainer]);

    useEffect(() => {
        // Определяем ширину контейнера для направлений и кнопки "еще"
        if (!isSuburbanWithDirections) {
            return;
        }

        calculateWidthContainerDirections();
    }, [calculateWidthContainerDirections, isSuburbanWithDirections]);

    useEffect(() => {
        // Ставим обработчик на вычисление ширины контейнера, при ресайзах окна
        if (!isSuburbanWithDirections) {
            return;
        }

        const throttledCalculateWidthContainerDirections = throttle(
            calculateWidthContainerDirections,
            200,
        );

        window.addEventListener(
            'resize',
            throttledCalculateWidthContainerDirections,
        );

        return () => {
            throttledCalculateWidthContainerDirections.cancel();
            window.removeEventListener(
                'resize',
                throttledCalculateWidthContainerDirections,
            );
        };
    }, [calculateWidthContainerDirections, isSuburbanWithDirections]);

    useEffect(() => {
        // Вычисляем ширину кнопки "еще"
        if (!refMoreDirectionsContainer.current) {
            return;
        }

        const widthMore = refMoreDirectionsContainer.current.offsetWidth + 1; // Компенсируем округление offsetWidth браузера до целых

        if (widthMoreDirections !== widthMore) {
            setWidthMoreDirections(widthMore);
        }
    }, [widthMoreDirections]);

    useEffect(() => {
        // Финальные вычисления
        if (
            !widthDirectionsContainer ||
            !widthMoreDirections ||
            !directionsWithWidth.directionsWithWidth
        ) {
            // Если хоть какие-то параметры еще не известны, то выходим
            return;
        }

        const newSortedDirections = getSortedDirections(
            directionsWithWidth,
            widthDirectionsContainer,
            widthMoreDirections,
            directionCode,
        );

        let totalWidth = 0;
        const availableWidthForDirections =
            widthDirectionsContainer - widthMoreDirections;
        const newMoreDirections: IDirection[] = [];
        const newVisibleDirections: IDirection[] = [];

        newSortedDirections.forEach(({code, title, width}) => {
            totalWidth += width;

            if (totalWidth <= availableWidthForDirections) {
                newVisibleDirections.push({code, title});
            }

            if (totalWidth > availableWidthForDirections) {
                newMoreDirections.push({code, title});
            }
        });

        if (!isEqual(sortedDirections, newVisibleDirections)) {
            setSortedDirections(newVisibleDirections);
        }

        if (!isEqual(moreDirections, newMoreDirections)) {
            setMoreDirections(newMoreDirections);
        }
    }, [
        directionCode,
        directionsWithWidth,
        moreDirections,
        sortedDirections,
        widthDirectionsContainer,
        widthMoreDirections,
    ]);

    const onClickMoreDirections = useCallback(() => {
        dispatch(setMoreDirectionsIsOpened(!moreDirectionsIsOpened));
    }, [dispatch, moreDirectionsIsOpened]);

    const onClickOutside = useCallback(() => {
        dispatch(setMoreDirectionsIsOpened(false));
    }, [dispatch]);

    const directionTabs = useCallback(
        (directionList?: IDirection[]) => {
            const isPlane = currentSubtype === StationSubtype.plane;
            const isTrain = currentSubtype === StationSubtype.train;
            const isTablo = currentSubtype === StationSubtype.tablo;

            const directionsData: {
                selected: boolean;
                key: string;
                href: string;
                text: string;

                onClick?: () => void;
            }[] =
                isTrain || isTablo || isPlane
                    ? [
                          {
                              selected: event === StationEventList.departure,
                              key: StationEventList.departure,
                              href: stationUrl({
                                  ...commonUrlParameters,
                                  event: StationEventList.departure,
                              }),
                              text: isPlane
                                  ? stationKeyset('departure-plane')
                                  : stationKeyset('departure'),
                          },
                          {
                              selected: event === StationEventList.arrival,
                              key: StationEventList.arrival,
                              href: stationUrl({
                                  ...commonUrlParameters,
                                  event: StationEventList.arrival,
                              }),
                              text: isPlane
                                  ? stationKeyset('arrival-plane')
                                  : stationKeyset('arrival'),
                          },
                      ]
                    : directionList
                    ? directionList.map(direction => ({
                          selected: direction.code === directionCode,
                          key: direction.code,
                          href: stationUrl({
                              ...commonUrlParameters,
                              direction: direction.code,
                          }),
                          text: direction.title,
                          onClick: getOnDirectionLinkClick(direction),
                      }))
                    : [];

            return directionsData.map(direction => (
                <Link
                    className={b('direction', {selected: direction.selected})}
                    key={direction.key}
                    href={direction.href}
                    onClick={direction.onClick}
                    colors="textPrimary"
                >
                    {upperFirst(direction.text)}
                </Link>
            ));
        },
        [commonUrlParameters, currentSubtype, directionCode, event],
    );

    const onClickLinkInMoreDirections = useCallback(() => {
        dispatch(setMoreDirectionsIsOpened(false));
    }, [dispatch]);

    const moreDirectionsElement = useCallback(
        (withPopup: boolean, ref?: MutableRefObject<HTMLDivElement | null>) => (
            <div className={b('moreDirectionsContainer')} ref={ref}>
                <span
                    className={b('moreDirectionsButton')}
                    onClick={onClickMoreDirections}
                >
                    {stationKeyset('more-directions')}

                    <Arrow
                        className={b('moreArrow')}
                        direction={moreDirectionsIsOpened ? 'up' : 'down'}
                        size="s"
                    />
                </span>

                {withPopup && (
                    <Popup
                        className={b('moreDirectionsPopup')}
                        visible={moreDirectionsIsOpened}
                        onClickOutside={onClickOutside}
                        theme="travel"
                        withoutArrow
                    >
                        <div className={b('moreDirectionsContent')}>
                            {moreDirections.map(({code, title}) => {
                                const href = stationUrl({
                                    ...commonUrlParameters,
                                    direction: code,
                                });

                                return (
                                    <Link
                                        className={b('moreDirection')}
                                        key={code}
                                        href={href}
                                        colors="inherit"
                                        onClick={onClickLinkInMoreDirections}
                                    >
                                        {upperFirst(title)}
                                    </Link>
                                );
                            })}
                        </div>
                    </Popup>
                )}
            </div>
        ),
        [
            commonUrlParameters,
            moreDirections,
            moreDirectionsIsOpened,
            onClickLinkInMoreDirections,
            onClickMoreDirections,
            onClickOutside,
        ],
    );

    return (
        <div className={b(undefined, className)}>
            <div className={b('allDirectionsContainer')}>
                {/* Видимый блок для показа пользователю */}
                <div
                    className={b('directionsContainer')}
                    ref={refDirectionsContainer}
                >
                    <div className={b('directions')}>
                        {directionTabs(sortedDirections).map(
                            (direction, index) => (
                                <div
                                    className={b('directionContainer')}
                                    key={index}
                                >
                                    {direction}
                                </div>
                            ),
                        )}
                    </div>

                    {moreDirections.length > 0 && moreDirectionsElement(true)}
                </div>

                {/* Невидимый блок для вычислений */}
                <div
                    className={b('fakeDirectionsContainer')}
                    ref={refDirectionsContainer}
                >
                    <div className={b('directions')} ref={refFakeDirections}>
                        {directionTabs(directions).map((direction, index) => (
                            <div
                                className={b('directionContainer')}
                                key={index}
                            >
                                {direction}
                            </div>
                        ))}
                    </div>

                    {moreDirectionsElement(false, refMoreDirectionsContainer)}
                </div>
            </div>

            {currentSubtype === StationSubtype.suburban && (
                <div className={b('suburbanArrivalContainer')}>
                    <Link
                        className={b('suburbanArrival', {
                            selected: event === StationEventList.arrival,
                        })}
                        href={stationUrl({
                            ...commonUrlParameters,
                            event: StationEventList.arrival,
                        })}
                        onClick={onSuburbanArrivalClick}
                        colors="textPrimary"
                    >
                        {stationKeyset('arrival')}
                    </Link>
                </div>
            )}
        </div>
    );
}

// Возвращает направления в таком порядке, чтобы выбранный элемент всегда был видимым
function getSortedDirections(
    directionsWithWidthObj: DirectionsWithWidth,
    widthDirectionsContainer: number,
    widthMoreDirections: number,
    directionCode?: string,
): DirectionWithWidth[] {
    const {directionsWithWidth} = directionsWithWidthObj;

    if (!directionsWithWidth) {
        return [];
    }

    if (!directionCode) {
        return directionsWithWidth;
    }

    const selectedDirectionIndex = directionsWithWidth.findIndex(
        ({code}) => code === directionCode,
    );

    if (selectedDirectionIndex === -1) {
        return directionsWithWidth;
    }

    const availableWidthForDirections =
        widthDirectionsContainer - widthMoreDirections;

    const widthDirectionsWithSeleted = directionsWithWidth
        .slice(0, selectedDirectionIndex + 1)
        .reduce((summ, {width}) => (summ += width), 0);

    if (widthDirectionsWithSeleted <= availableWidthForDirections) {
        return directionsWithWidth;
    }

    const directionsWithTotalWidthWithoutSelectedDirection = directionsWithWidth
        .slice(0, selectedDirectionIndex)
        .concat(...directionsWithWidth.slice(selectedDirectionIndex + 1))
        .reduce<{totalWidth: number; directions: DirectionWithTotalWidth[]}>(
            (result, direction) => {
                result.totalWidth += direction.width;

                result.directions.push({
                    ...direction,
                    totalWidth: result.totalWidth,
                });

                return result;
            },
            {totalWidth: 0, directions: []},
        );

    let positionForInsert = 0;

    const selectedDirection = directionsWithWidth[selectedDirectionIndex];
    const selectedDirectionWidth = selectedDirection.width;
    const {directions} = directionsWithTotalWidthWithoutSelectedDirection;

    while (
        selectedDirectionWidth + directions[positionForInsert].totalWidth <=
        availableWidthForDirections
    ) {
        positionForInsert++;
    }

    const sortedDirections =
        directionsWithTotalWidthWithoutSelectedDirection.directions.map(
            ({code, title, width}) => ({code, title, width}),
        );

    sortedDirections.splice(positionForInsert, 0, selectedDirection);

    return sortedDirections;
}
