import moment from 'moment';
import React, {memo, useEffect} from 'react';
import B from 'bem-cn-lite';

import {HUMAN, YEAR} from '../../../lib/date/formats';

import StationSubtype from '../../../interfaces/state/station/StationSubtype';
import StationDateSpecialValue from '../../../interfaces/date/StationDateSpecialValue';
import StationEventList from '../../../interfaces/state/station/StationEventList';
import StationType from '../../../interfaces/state/station/StationType';

import makeUrl from '../../../lib/url/makeUrl';
import insertMarkupIntoKey from '../../../i18n/_utils/insertMarkupIntoKey';
import {reachGoalOnce, reachGoal} from '../../../lib/yaMetrika';
import useSelector from '../../useSelector';

import Link from '../../../components/Link';

import stationKeyset from '../../../i18n/station';
import stationPlaneKeyset from '../../../i18n/station-plane-seo';

const b = B('DisclaimerText');

function stationInfoClick(): void {
    reachGoal('station_info_link_click');
}

interface IDisclaimerText {
    className?: string;
}

export default memo(DisclaimerText);

function DisclaimerText({
    className,
}: IDisclaimerText): React.ReactElement | null {
    const title = useSelector(state => state.station.title);
    const hasPopularTitle = useSelector(state => state.station.hasPopularTitle);
    const type = useSelector(state => state.station.type);
    const subtype = useSelector(state => state.station.currentSubtype);
    const titleGenitive = useSelector(state => state.station.titleGenitive);
    const fullTitle = useSelector(state => state.station.fullTitle);
    const fullTitleDative = useSelector(state => state.station.fullTitleDative);
    const fullTitleGenitive = useSelector(
        state => state.station.fullTitleGenitive,
    );
    const whenDate = useSelector(state => state.station.whenDate);
    const whenSpecial = useSelector(state => state.station.whenSpecial);
    const event = useSelector(state => state.station.event);
    const terminals = useSelector(state => state.station.terminals);
    const terminalName = useSelector(state => state.station.terminalName);
    const id = useSelector(state => state.station.id);
    const now = useSelector(state => state.station.now);
    const scheduleBlocks = useSelector(state => state.station.scheduleBlocks);
    const tld = useSelector(state => state.tld);
    const language = useSelector(state => state.language);

    const terminal = terminals.find(
        terminalObj => terminalObj.name === terminalName,
    );
    const year = moment.parseZone(whenDate ?? now).format(YEAR);
    let content: React.ReactElement | null = null;

    switch (subtype) {
        case StationSubtype.suburban:
            content =
                whenSpecial === StationDateSpecialValue.allDays ? (
                    <p className={b('text')}>
                        {stationKeyset('disclaimer-suburban-all-days', {
                            title: fullTitleDative || fullTitle || title,
                        })}
                    </p>
                ) : (
                    <p className={b('text')}>
                        {stationKeyset('disclaimer-suburban', {
                            title: fullTitleDative || fullTitle || title,
                            date: moment(whenDate).format(HUMAN),
                            year,
                        })}
                    </p>
                );
            break;
        case StationSubtype.train:
            content = (
                <>
                    <h2 className={b('title')}>
                        {hasPopularTitle
                            ? stationKeyset('disclaimer-train-title-popular', {
                                  titleGenitive:
                                      fullTitleGenitive ||
                                      titleGenitive ||
                                      fullTitle ||
                                      title,
                              })
                            : stationKeyset('disclaimer-train-title', {
                                  titleDative:
                                      fullTitleDative || fullTitle || title,
                              })}
                    </h2>
                    <p className={b('text')}>
                        {stationKeyset('disclaimer-train', {
                            titleGenitive:
                                fullTitleGenitive ||
                                titleGenitive ||
                                fullTitle ||
                                title,
                            year,
                        })}
                    </p>
                </>
            );
            break;
        case StationSubtype.plane:
            if (!terminal) {
                const askForInformationPhonesText = (
                    // Из-за особенностей insertMarkupIntoKey для передаваемых элементов необходимо задать key
                    <Link
                        key="askForInformationPhonesText"
                        href={makeUrl(`/info/station/${id}`, tld, language)}
                        onClick={stationInfoClick}
                    >
                        {stationPlaneKeyset('ask-for-information-phones')}
                    </Link>
                );

                content = (
                    <>
                        <p className={b('text', {isPlane: true})}>
                            {event === StationEventList.arrival
                                ? stationPlaneKeyset(
                                      'disclaimer-arrival-first-part',
                                      {
                                          airportTitle: title,
                                          today:
                                              whenSpecial ===
                                              StationDateSpecialValue.today,
                                          date:
                                              whenSpecial !==
                                              StationDateSpecialValue.allDays
                                                  ? moment(whenDate).format(
                                                        HUMAN,
                                                    )
                                                  : '',
                                      },
                                  )
                                : stationPlaneKeyset(
                                      'disclaimer-departure-first-part',
                                      {
                                          airportTitle: title,
                                          today:
                                              whenSpecial ===
                                              StationDateSpecialValue.today,
                                          date:
                                              whenSpecial !==
                                              StationDateSpecialValue.allDays
                                                  ? moment(whenDate).format(
                                                        HUMAN,
                                                    )
                                                  : '',
                                      },
                                  )}
                        </p>

                        <p className={b('text', {isPlane: true})}>
                            {stationPlaneKeyset('disclaimer-second-part-list')}
                        </p>

                        <ul className={b('list')}>
                            <li>
                                {stationPlaneKeyset(
                                    'disclaimer-second-part-list-item-1',
                                )}
                            </li>
                            <li>
                                {stationPlaneKeyset(
                                    'disclaimer-second-part-list-item-2',
                                )}
                            </li>
                            <li>
                                {stationPlaneKeyset(
                                    'disclaimer-second-part-list-item-3',
                                )}
                            </li>
                            <li>
                                {stationPlaneKeyset(
                                    'disclaimer-second-part-list-item-4',
                                )}
                            </li>
                        </ul>

                        <p className={b('text', {isPlane: true})}>
                            {stationPlaneKeyset('disclaimer-third-part')}
                        </p>

                        <p className={b('text', {isPlane: true})}>
                            {insertMarkupIntoKey(
                                stationPlaneKeyset,
                                'disclaimer-fourth-part',
                                {askForInformationPhonesText},
                            )}
                        </p>
                    </>
                );
            } else {
                const contactInformationPhonesText = (
                    // Из-за особенностей insertMarkupIntoKey для передаваемых элементов необходимо задать key
                    <Link
                        key="contactInformationPhonesText"
                        href={makeUrl(`/info/station/${id}}`, tld, language)}
                        onClick={stationInfoClick}
                    >
                        {stationPlaneKeyset('contact-information-phones')}
                    </Link>
                );

                content = (
                    <>
                        <p className={b('text', {isPlane: true})}>
                            {event === StationEventList.arrival
                                ? stationPlaneKeyset(
                                      'disclaimer-arrival-with-terminal-first-part',
                                      {
                                          isDomestic: Boolean(
                                              terminal.isDomestic,
                                          ),
                                          isInternational: Boolean(
                                              terminal.isInternational,
                                          ),
                                          noSign:
                                              !terminal.isDomestic &&
                                              !terminal.isInternational,
                                          terminal: terminal.name,
                                          airportTitle: title,
                                          date: moment(whenDate).format(HUMAN),
                                      },
                                  )
                                : stationPlaneKeyset(
                                      'disclaimer-departure-with-terminal-first-part',
                                      {
                                          isDomestic: Boolean(
                                              terminal.isDomestic,
                                          ),
                                          isInternational: Boolean(
                                              terminal.isInternational,
                                          ),
                                          terminal: terminal.name,
                                          airportTitle: title,
                                          date: moment(whenDate).format(HUMAN),
                                      },
                                  )}
                        </p>
                        <p className={b('text', {isPlane: true})}>
                            {insertMarkupIntoKey(
                                stationPlaneKeyset,
                                'disclaimer-with-terminal-second-part',
                                {contactInformationPhonesText},
                            )}
                        </p>
                    </>
                );
            }

            break;
        case StationSubtype.tablo:
            content = (
                <>
                    <h2 className={b('title')}>
                        {hasPopularTitle
                            ? stationKeyset('disclaimer-tablo-title-popular', {
                                  titleGenitive:
                                      fullTitleGenitive ||
                                      titleGenitive ||
                                      fullTitle ||
                                      title,
                              })
                            : stationKeyset('disclaimer-tablo-title', {
                                  titleDative:
                                      fullTitleDative || fullTitle || title,
                              })}
                    </h2>
                    <p className={b('text')}>
                        {stationKeyset('disclaimer-tablo', {
                            titleGenitive:
                                fullTitleGenitive ||
                                titleGenitive ||
                                fullTitle ||
                                title,
                            year,
                        })}
                    </p>
                </>
            );
            break;
    }

    switch (type) {
        case StationType.bus:
            {
                if (scheduleBlocks.length === 0) {
                    break;
                }

                const stationName = fullTitle ?? title;

                const disclaimerAttentionArr = stationKeyset(
                    'disclaimer-bus-attention',
                ).split('%%');

                content = (
                    <>
                        <p className={b('text')}>
                            {stationKeyset('disclaimer-bus', {
                                title: stationName,
                                whenSpecial: whenSpecial ?? '',
                                whenDate: whenDate
                                    ? moment(whenDate).format(HUMAN)
                                    : '',
                            })}
                        </p>

                        <p className={b('text')}>
                            {disclaimerAttentionArr[0]}
                            <Link
                                href={makeUrl(
                                    `/info/station/${id}`,
                                    tld,
                                    language,
                                )}
                            >
                                {disclaimerAttentionArr[1]}
                            </Link>
                            {disclaimerAttentionArr[2]}
                        </p>
                    </>
                );
            }

            break;
        case StationType.water:
            {
                if (scheduleBlocks.length === 0) {
                    break;
                }

                const stationNameGenitive =
                    fullTitleGenitive ?? fullTitle ?? title;

                content = (
                    <p className={b('text')}>
                        {stationKeyset('disclaimer-water', {
                            titleGenitive: stationNameGenitive,
                            year,
                        })}
                    </p>
                );
            }

            break;
    }

    useEffect(() => {
        if (subtype === StationSubtype.plane && content) {
            reachGoalOnce('station_info_link_shown');
        }
    }, [content, subtype]);

    if (!content) {
        return null;
    }

    return <div className={b(undefined, className)}>{content}</div>;
}
