import React, {memo, ReactElement, useEffect} from 'react';
import B from 'bem-cn-lite';

import StationSubtype from '../../../interfaces/state/station/StationSubtype';

import getTitle from './getTitle';
import {stationUrl} from '../../../lib/url/stationUrl';
import {reachGoal, reachGoalOnce} from '../../../lib/yaMetrika';
import useSelector from '../../useSelector';
import {makeCacheable} from '../../../lib/cache';

import Link from '../../Link';
import StationWayToAirport from '../../StationWayToAirport/StationWayToAirport';

const b = B('PageSubtypeTabs');

interface ISubtypeLinkClickCallback {
    (): void;
}

const getSubtypeLinkClickCallback = makeCacheable(
    (subtype: StationSubtype): ISubtypeLinkClickCallback =>
        () => {
            reachGoal(`station_alternative_type_${subtype}_click`);
        },
);

interface IPageSubtypeTabs {
    className?: string;
}

export default memo(PageSubtypeTabs);

function PageSubtypeTabs({className}: IPageSubtypeTabs): ReactElement {
    const stationId = useSelector(state => state.station.id);
    const subtypes = useSelector(state => state.station.subtypes);
    const currentSubtype = useSelector(state => state.station.currentSubtype);
    const type = useSelector(state => state.station.type);
    const mainSubtype = useSelector(state => state.station.mainSubtype);
    const whenDate = useSelector(state => state.station.whenDate);
    const whenSpecial = useSelector(state => state.station.whenSpecial);
    const tld = useSelector(state => state.tld);
    const language = useSelector(state => state.language);

    useEffect(() => {
        subtypes.forEach(subtype => {
            if (subtype !== currentSubtype) {
                reachGoalOnce(`station_alternative_type_${subtype}_shown`);
            }
        });
    });

    const wayToAirportLink = <StationWayToAirport linkColor="textSecondary" />;

    return (
        <div className={b({}, className)}>
            {subtypes.map(subtype => {
                const isLink = subtype !== currentSubtype;

                const href = stationUrl({
                    id: stationId,
                    tld,
                    language,
                    subtype,
                    mainSubtype,
                    isMobile: true,
                    date: whenDate,
                    special: whenSpecial,
                    type,
                });

                const title = getTitle(subtype);

                return (
                    title && (
                        <span
                            className={b('tab', {selected: !isLink})}
                            key={subtype}
                        >
                            {href && isLink ? (
                                <Link
                                    href={href}
                                    className="tabLink"
                                    colors="textSecondary"
                                    onClick={getSubtypeLinkClickCallback(
                                        subtype,
                                    )}
                                >
                                    {title}
                                </Link>
                            ) : (
                                title
                            )}
                        </span>
                    )
                );
            })}

            <span className={b('tab')}>{wayToAirportLink}</span>
        </div>
    );
}
