import React, {useMemo} from 'react';
import B from 'bem-cn-lite';

import {STATION_POPULAR_DIRECTIONS_LIMIT} from '../../../lib/station/stationConstants';
import {CHAR_EM_DASH} from '../../../lib/stringUtils';

import IPopularDirections from '../../../interfaces/state/station/IStationPopularDirectionsFromApi';
import IPopularDirectionStation from '../../../interfaces/state/station/IPopularDirectionStation';
import StationEventList from '../../../interfaces/state/station/StationEventList';

import useSelector from '../../useSelector';
import getCanonicalSearchUrl from '../../../lib/url/getCanonicalSearchUrl';

import Link from '../../Link';

import stationKeyset from '../../../i18n/station';

const b = B('StationPopularDirections');
const isMobile = process.env.PLATFORM === 'mobile';

function getTitle(
    station: IPopularDirectionStation,
    stationFromPopularDirections: IPopularDirectionStation,
    insertAfterStationName: boolean,
): string {
    return insertAfterStationName
        ? `${station.title} ${CHAR_EM_DASH} ${stationFromPopularDirections.title}`
        : `${stationFromPopularDirections.title} ${CHAR_EM_DASH} ${station.title}`;
}

interface IPopularDirectionsProps {
    event: StationEventList;

    popularDirections?: IPopularDirections;
    title?: string;
    className?: string;
}

export default React.memo(PopularDirections);

function PopularDirections({
    event,

    popularDirections,
    title,
    className,
}: IPopularDirectionsProps): React.ReactElement | null {
    if (
        !popularDirections ||
        (!popularDirections.from.length && !popularDirections.to.length)
    ) {
        return null;
    }

    const language = useSelector(state => state.language);
    const tld = useSelector(state => state.tld);

    const popularDirectionsList = useMemo(() => {
        const {searchTransportType, station, to, from} = popularDirections;
        const result: React.ReactElement[] = [];
        const isDeparture = event === StationEventList.departure;
        const [primary, secondary] = isDeparture ? [to, from] : [from, to];

        for (let i = 0; i < STATION_POPULAR_DIRECTIONS_LIMIT; i++) {
            if (i < primary.length) {
                const stationPrimary = primary[i];

                result.push(
                    <div
                        className={b('direction')}
                        key={`primary${stationPrimary.slug}`}
                    >
                        <Link
                            className={b('directionLink')}
                            href={getCanonicalSearchUrl({
                                transportType: searchTransportType,
                                toSlug: isDeparture
                                    ? stationPrimary.slug
                                    : station.slug,
                                fromSlug: isDeparture
                                    ? station.slug
                                    : stationPrimary.slug,
                                tld,
                                language,
                            })}
                        >
                            {getTitle(station, stationPrimary, isDeparture)}
                        </Link>
                    </div>,
                );
            } else if (i - primary.length < secondary.length) {
                const stationSecondary = secondary[i - primary.length];

                result.push(
                    <div
                        className={b('direction')}
                        key={`secondary${stationSecondary.slug}`}
                    >
                        <Link
                            className={b('directionLink')}
                            href={getCanonicalSearchUrl({
                                transportType: searchTransportType,
                                toSlug: isDeparture
                                    ? station.slug
                                    : stationSecondary.slug,
                                fromSlug: isDeparture
                                    ? stationSecondary.slug
                                    : station.slug,
                                tld,
                                language,
                            })}
                        >
                            {getTitle(station, stationSecondary, !isDeparture)}
                        </Link>
                    </div>,
                );
            }
        }

        return result;
    }, [event, language, popularDirections, tld]);

    return (
        <div className={b({isMobile}, className)}>
            <div className={b('title')}>
                {title || stationKeyset('popular-directions')}
            </div>
            <div className={b('directionsContainer')}>
                {popularDirectionsList}
            </div>
        </div>
    );
}
