import React, {
    memo,
    useMemo,
    useCallback,
    ReactElement,
    MouseEvent,
    useEffect,
} from 'react';
import B from 'bem-cn-lite';
import page from 'page';
import moment from 'moment';
import lowerFirst from 'lodash/lowerFirst';

import {HUMAN, RU_LOCALE_DATE} from '../../../lib/date/formats';

import DateSpecialValue from '../../../interfaces/date/DateSpecialValue';
import StationSubtype from '../../../interfaces/state/station/StationSubtype';
import PrimaryPosition from '../../../interfaces/lib/dimensions/PrimaryPosition';
import SecondaryPosition from '../../../interfaces/lib/dimensions/SecondaryPosition';
import StationDateSpecialValue from '../../../interfaces/date/StationDateSpecialValue';
import TimeOption from './interfaces/TimeOption';
import TerminalOption from './interfaces/TerminalOption';
import IWhen from '../../../interfaces/state/IWhen';
import StationTime from '../../../interfaces/state/station/StationTime';
import IconGlyph from '../../../interfaces/components/IconGlyph';

import useSelector from '../../useSelector';
import useDispatch from '../../useDispatch';
import {reachGoal, reachGoalOnce} from '../../../lib/yaMetrika';
import {stationUrl} from '../../../lib/url/stationUrl';
import {buildFromObject} from '../../../lib/date/build';
import {
    getRange,
    getToday,
    getParseParams,
    extendRangeToArray,
} from '../../../lib/date/utils';
import getValueFromEnum from '../../../lib/enum/getValueFromEnum';
import timeOptions from './timeOptions';
import formatTitleTimes from './formatTitleTimes';
import formatTitleTerminals from './formatTitleTerminals';
import getTerminalOptions from './getTerminalOptions';

import {
    setCalendarIsOpened,
    changeFilterTerminalName,
    changeFilterTime,
} from '../../../actions/station';

import Icon from '../../Icon/Icon';
import Button2 from '../../Button2/Button2';
import Select from '../../basic/Select/Select';
import Popup from '../../../components/Popup/Popup';
import CalendarScroller from '../../../components/CalendarScroller/CalendarScroller';
import CalendarToolbar from '../../CalendarToolbar/CalendarToolbar.desktop';
import StationCalendarDayLink from '../../../components/StationCalendarDayLink/StationCalendarDayLink';
import StationDatePicker from '../../StationDatePicker';

import searchFormKeyset from '../../../i18n/search-form';
import stationKeyset from '../../../i18n/station';

const b = B('StationFilters');

const popupPositions: [PrimaryPosition, SecondaryPosition][] = [
    [PrimaryPosition.bottom, SecondaryPosition.left],
    [PrimaryPosition.bottom, SecondaryPosition.right],
];

interface IStationFilters {
    className?: string;
}

export default memo(StationFilters);

function StationFilters({className}: IStationFilters): ReactElement {
    const tld = useSelector(state => state.tld);
    const station = useSelector(state => state.station);
    const language = useSelector(state => state.language);
    const nowTime = useSelector(state => state.searchForm.time);

    const dispatch = useDispatch();

    const {
        id,
        type,
        event,
        whenDate,
        whenSpecial,
        directionCode,
        currentSubtype,
        mainSubtype,
        terminals,
        terminalName,
        time,
        calendarIsOpened,
    } = station;

    const onSpecialDateClick = useCallback(
        (special: DateSpecialValue) => {
            reachGoal(`station_date_${special}_click`);
            page.show(
                stationUrl({
                    special: getValueFromEnum(special, StationDateSpecialValue),
                    id,
                    subtype: currentSubtype,
                    mainSubtype,
                    event,
                    direction: directionCode,
                    terminalName,
                    type,
                    language,
                    tld,
                    time,
                }),
            );
        },
        [
            id,
            currentSubtype,
            mainSubtype,
            terminalName,
            event,
            directionCode,
            type,
            language,
            tld,
            time,
        ],
    );

    const onCalendarToolbarClick = useCallback(
        (e: MouseEvent<HTMLElement>) => {
            const {dataset} = e.currentTarget;
            const {date, special, weekday} = dataset;
            const value = buildFromObject(
                {
                    date,
                    special,
                    weekday,
                },
                getParseParams({time: nowTime, language, tld}),
            );

            onSpecialDateClick(value.special);

            dispatch(setCalendarIsOpened(false));
        },
        [dispatch, nowTime, language, tld, onSpecialDateClick],
    );

    const onCalendarClick = useCallback(() => {
        reachGoal('station_date_specific_date_click');
        dispatch(setCalendarIsOpened(false));
    }, [dispatch]);

    const onCalendarClickOutside = useCallback(() => {
        dispatch(setCalendarIsOpened(false));
    }, [dispatch]);

    const terminalOptions: TerminalOption[] = useMemo(
        () => getTerminalOptions(terminals),
        [terminals],
    );

    const selectedTerminalOption = terminalOptions.find(
        terminal => terminal.value === terminalName,
    );

    const onChangeTerminal = useCallback(
        (option: TerminalOption) => {
            reachGoal('terminal_selected_desktop', {
                terminalName: option.value,
            });
            dispatch(changeFilterTerminalName(option.value));
        },
        [dispatch],
    );

    const selectedTimeOption = timeOptions.find(({value}) => value === time);

    const onChangeTime = useCallback(
        (option: TimeOption) => {
            reachGoal('time_selected_desktop', {timeRange: option.value});
            dispatch(changeFilterTime(option.value));
        },
        [dispatch],
    );

    const onDatepickerClick = useCallback(() => {
        dispatch(setCalendarIsOpened(!calendarIsOpened));
    }, [dispatch, calendarIsOpened]);

    const isPlane = currentSubtype === StationSubtype.plane;

    useEffect(() => {
        if (terminalOptions.length) {
            reachGoalOnce('terminal_filter_shown_desktop');
        }

        if (isPlane) {
            reachGoalOnce('time_filter_shown_desktop');
        }
    }, [terminalOptions, isPlane]);

    // Конструируем костыльный объект для value календаря, потому что он заточен только на форму данных поиска
    const datePickerValue: IWhen = useMemo(() => {
        return {
            text: '',
            hint: '',
            formatted: '',
            shortFormatted: '',

            date: whenDate,
            special:
                whenSpecial && getValueFromEnum(whenSpecial, DateSpecialValue),
        };
    }, [whenDate, whenSpecial]);

    const calendarPopup = useMemo(() => {
        const today = getToday(nowTime);
        const range = getRange(today);
        const months = extendRangeToArray(range, 'month', 'month');

        return (
            <Popup
                className={b('calendarPopup')}
                contentClassName={b('calendarPopupContent')}
                visible={calendarIsOpened}
                positions={popupPositions}
                onClickOutside={onCalendarClickOutside}
                withoutArrow
            >
                {isPlane && (
                    <CalendarToolbar
                        className={b('toolbar')}
                        value={datePickerValue}
                        language={language}
                        onClick={onCalendarToolbarClick}
                    />
                )}

                <CalendarScroller
                    range={range}
                    today={today}
                    value={datePickerValue}
                    months={months}
                    time={nowTime}
                    language={language}
                    onClick={onCalendarClick}
                    DayComponent={StationCalendarDayLink}
                />
            </Popup>
        );
    }, [
        nowTime,
        calendarIsOpened,
        onCalendarClickOutside,
        isPlane,
        datePickerValue,
        language,
        onCalendarToolbarClick,
        onCalendarClick,
    ]);

    const formatted = moment(whenDate).format(HUMAN).toLowerCase();
    const convertedWhenSpecial = whenSpecial
        ? getValueFromEnum(whenSpecial, DateSpecialValue)
        : undefined;

    const getFilters = useMemo(() => {
        if (isPlane) {
            return (
                <>
                    {terminalOptions.length > 0 && (
                        <Select
                            className={b('terminalsSelect')}
                            options={terminalOptions}
                            value={selectedTerminalOption}
                            onChange={onChangeTerminal}
                            formatTitle={formatTitleTerminals}
                            type="grayButton"
                            popupPositions={popupPositions}
                        />
                    )}

                    <Button2
                        className={b('dateButton')}
                        theme="normal"
                        onClick={onDatepickerClick}
                        iconRight={
                            <Icon
                                glyph={IconGlyph.calendar}
                                className={b('calendarIcon')}
                            />
                        }
                    >
                        {whenSpecial
                            ? searchFormKeyset(whenSpecial)
                            : formatted}
                    </Button2>

                    {calendarPopup}

                    <Select
                        className={b('timeSelect')}
                        options={timeOptions}
                        value={selectedTimeOption}
                        onChange={onChangeTime}
                        formatTitle={formatTitleTimes}
                        type="grayButton"
                        popupPositions={popupPositions}
                    />
                </>
            );
        }

        return (
            <>
                <StationDatePicker
                    whenSpecial={convertedWhenSpecial}
                    whenDate={whenDate}
                    onSpecialDateClick={onSpecialDateClick}
                    onDatepickerClick={onDatepickerClick}
                />

                {calendarPopup}
            </>
        );
    }, [
        isPlane,
        convertedWhenSpecial,
        whenDate,
        onSpecialDateClick,
        onDatepickerClick,
        calendarPopup,
        terminalOptions,
        selectedTerminalOption,
        onChangeTerminal,
        whenSpecial,
        formatted,
        selectedTimeOption,
        onChangeTime,
    ]);

    const dateText =
        whenSpecial === StationDateSpecialValue.allDays
            ? stationKeyset('schedule-for-print-on-all-days')
            : stationKeyset('schedule-for-print', {
                  date: moment(whenDate).format(RU_LOCALE_DATE),
              });

    const timeRangeText =
        selectedTimeOption && selectedTimeOption.value !== StationTime.all
            ? `, ${selectedTimeOption.title.toLowerCase()}`
            : undefined;

    let terminalText;

    if (selectedTerminalOption?.value) {
        const terminalValue = stationKeyset('terminal', {
            terminalName: selectedTerminalOption.title,
        });

        terminalText = `, ${lowerFirst(terminalValue)}`;
    }

    return (
        <div className={b(undefined, className)}>
            {getFilters}

            <div className={b('stationFiltersForPrint')}>
                {`${dateText}${timeRangeText ?? ''}${terminalText ?? ''}`}
            </div>
        </div>
    );
}
