import React, {
    memo,
    ReactElement,
    useCallback,
    useMemo,
    useEffect,
} from 'react';
import B from 'bem-cn-lite';
import page from 'page';
import moment from 'moment';
import upperFirst from 'lodash/upperFirst';

import {HUMAN} from '../../../lib/date/formats';
import {DIRECTION_CODE_ALL} from '../../../lib/station/stationConstants';

import StationType from '../../../interfaces/state/station/StationType';
import StationSubtype from '../../../interfaces/state/station/StationSubtype';
import StationEventList from '../../../interfaces/state/station/StationEventList';
import StationDateSpecialValue from '../../../interfaces/date/StationDateSpecialValue';
import TimeOption from './interfaces/TimeOption';
import TerminalOption from './interfaces/TerminalOption';
import IconGlyph from '../../../interfaces/components/IconGlyph';
import IStopSuggestElement from '../../../interfaces/lib/station/IStopSuggestElement';

import useDispatch from '../../useDispatch';
import useSelector from '../../useSelector';
import {reachGoal, reachGoalOnce} from '../../../lib/yaMetrika';
import {stationUrl} from '../../../lib/url/stationUrl';
import isDateRobot from '../../../lib/date/isDateRobot';
import getValueFromEnum from '../../../lib/enum/getValueFromEnum';
import timeOptions from './timeOptions';
import formatTitleTerminals from './formatTitleTerminals';
import getTerminalOptions from './getTerminalOptions';
import formatTitleTimes from './formatTitleTimes';
import getFunctionForGetDataStationSuggest from '../../../lib/station/getFunctionForGetDataStationSuggest';
import getRegexpForSuggestStops from '../../../lib/station/getRegexpForSuggestStops';

import {
    setCalendarIsOpened,
    changeFilterTime,
    changeFilterTerminalName,
    setStationFilterIsOpened,
    setStopText,
    changeFilterStop,
} from '../../../actions/station';

import Icon from '../../Icon/Icon';
import GlobalPopup from '../../GlobalPopup';
import Button2 from '../../Button2/Button2';
import Select from '../../basic/Select/Select';
import NativeHorizontScroll from '../../NativeHorizontScroll/NativeHorizontScroll';
import Calendar from '../../Calendar/Calendar.mobile';
import SuggestMobile from '../../SuggestMobile/SuggestMobile';

import stationKeyset from '../../../i18n/station';

const b = B('StationFilters');

type DirectionOption = {
    title: string;
    value: string;

    event?: StationEventList;
};

interface IStationFilters {
    className?: string;
}

export default memo(StationFilters);

function StationFilters({className}: IStationFilters): ReactElement {
    const tld = useSelector(state => state.tld);
    const language = useSelector(state => state.language);
    const nowTime = useSelector(state => state.searchForm.time);
    const station = useSelector(state => state.station);
    const {
        id,
        type,
        currentSubtype,
        mainSubtype,
        whenDate,
        whenSpecial,
        event,
        directions,
        directionCode,
        time,
        terminals,
        terminalName,
        stops,
        stopText,
        stationFilterIsOpened,
        calendarIsOpened,
    } = station;

    const onChangeDirection = useCallback(
        (option: DirectionOption) => {
            const url = stationUrl({
                id,
                tld,
                language,
                type,
                subtype: currentSubtype,
                mainSubtype,
                isMobile: true,
                date: whenDate,
                special: whenSpecial,
                event: option.event,
                direction: !option.event ? option.value : undefined,
            });

            if (currentSubtype === StationSubtype.suburban) {
                if (option.event) {
                    reachGoal(
                        `station_${currentSubtype}_${option.event}_click`,
                    );
                } else {
                    reachGoal(
                        `station_direction_${
                            option.value === DIRECTION_CODE_ALL
                                ? DIRECTION_CODE_ALL
                                : 'specific'
                        }_click`,
                    );
                }
            }

            page.show(url);
        },
        [
            currentSubtype,
            id,
            language,
            mainSubtype,
            tld,
            type,
            whenDate,
            whenSpecial,
        ],
    );

    const dispatch = useDispatch();

    const onClickDateButton = useCallback(() => {
        reachGoal('station_date_click');
        dispatch(setCalendarIsOpened(!calendarIsOpened));
    }, [calendarIsOpened, dispatch]);

    const onCloseCalendar = useCallback(() => {
        dispatch(setCalendarIsOpened(false));
    }, [dispatch]);

    const onClickSelectDirection = useCallback(() => {
        reachGoal('station_direction_click');
    }, []);

    const dateForCalendar = useMemo(() => ({date: whenDate}), [whenDate]);

    const terminalOptions: TerminalOption[] = useMemo(
        () => getTerminalOptions(terminals),
        [terminals],
    );

    const selectedTerminalOption = terminalOptions.find(
        terminal => terminal.value === terminalName,
    );

    const onChangeTerminal = useCallback(
        (option: TerminalOption) => {
            reachGoal('terminal_selected_mobile', {terminalName: option.value});
            dispatch(changeFilterTerminalName(option.value));
        },
        [dispatch],
    );

    const onChangeCalendar = useCallback(
        (e, value: {value: {date?: string; special?: string}}) => {
            const {date, special} = value.value;
            const specialAsStationDateSpecialValue = special
                ? getValueFromEnum(special, StationDateSpecialValue)
                : undefined;

            const url = stationUrl({
                id,
                tld,
                language,
                type,
                subtype: currentSubtype,
                mainSubtype,
                isMobile: true,
                date: date && isDateRobot(date) ? date : undefined,
                special: specialAsStationDateSpecialValue,
                event,
                terminalName,
                direction: directionCode,
                time,
            });

            reachGoal(`station_date_${special || 'specific_date'}_click`);

            page.show(url);
        },
        [
            currentSubtype,
            directionCode,
            event,
            terminalName,
            id,
            language,
            mainSubtype,
            tld,
            type,
            time,
        ],
    );

    const directionOptions: DirectionOption[] = (directions ?? []).map(
        ({title, code}) => ({
            title: upperFirst(title),
            value: code,
        }),
    );

    if (currentSubtype === StationSubtype.suburban) {
        directionOptions.push({
            title: stationKeyset('arrival'),
            value: StationEventList.arrival,
            event: StationEventList.arrival,
        });
    }

    if (
        currentSubtype === StationSubtype.train ||
        currentSubtype === StationSubtype.tablo
    ) {
        directionOptions.push({
            title: stationKeyset('departure'),
            value: StationEventList.departure,
            event: StationEventList.departure,
        });
        directionOptions.push({
            title: stationKeyset('arrival'),
            value: StationEventList.arrival,
            event: StationEventList.arrival,
        });
    }

    const directionValue = directionOptions.find(
        direction =>
            direction.value === directionCode || direction.event === event,
    );

    const onChangeTime = useCallback(
        (option: TimeOption) => {
            reachGoal('time_selected_mobile', {timeRange: option.value});
            dispatch(changeFilterTime(option.value));
        },
        [dispatch],
    );

    const onClickSearchButton = useCallback(() => {
        dispatch(setStationFilterIsOpened(true));
    }, [dispatch]);

    const onCloseStationSuggest = useCallback(() => {
        dispatch(setStationFilterIsOpened(false));
    }, [dispatch]);

    const onChangeStopText = useCallback(
        (value: string) => {
            dispatch(setStopText(value));
        },
        [dispatch],
    );

    const onClickStationSuggest = useCallback(
        (element: IStopSuggestElement) => {
            const stopInStore = stops.find(
                stopObj => stopObj.id === element.id,
            );

            if (stopInStore) {
                dispatch(setStationFilterIsOpened(false));
                dispatch(changeFilterStop(stopInStore));
            }
        },
        [dispatch, stops],
    );

    const getStationSuggest = useMemo(
        () => getFunctionForGetDataStationSuggest(stops, 20),
        [stops],
    );

    const regExpFroFilterSuggest = useMemo(
        () => getRegexpForSuggestStops(stopText),
        [stopText],
    );

    const selectedTimeOption = timeOptions.find(({value}) => value === time);

    let whenText: string | undefined;

    if (whenDate) {
        whenText = moment(whenDate).format(HUMAN);
    }

    switch (whenSpecial) {
        case StationDateSpecialValue.today:
            whenText = stationKeyset('scheduleToday');
            break;
        case StationDateSpecialValue.tomorrow:
            whenText = stationKeyset('scheduleTomorrow');
            break;
        case StationDateSpecialValue.allDays:
            whenText = stationKeyset('scheduleAllDays');
            break;
    }

    const isPlanePage = type === StationType.plane;
    const isBusWaterPage =
        type === StationType.water || type === StationType.bus;

    const otherElements: ReactElement[] = [];
    const buttons: ReactElement[] = [
        <Button2
            key="dateButton"
            className={b('dateButton')}
            theme="normal"
            onClick={onClickDateButton}
            iconRight={
                <Icon
                    glyph={IconGlyph.calendar}
                    className={b('calendarIcon')}
                />
            }
        >
            {whenText}
        </Button2>,
    ];

    if (directionOptions.length) {
        buttons.unshift(
            <Select
                className={b('directionsSelect')}
                key="directionsSelect"
                options={directionOptions}
                value={directionValue}
                onChange={onChangeDirection}
                onClick={onClickSelectDirection}
                listItemSize="big"
                type="grayButton"
                ellipsis
                minWidth={100}
            />,
        );
    }

    if (calendarIsOpened) {
        buttons.push(
            <GlobalPopup key="calendarPopup">
                <Calendar
                    time={nowTime}
                    value={dateForCalendar}
                    language={language}
                    fixed
                    opened={calendarIsOpened}
                    onClose={onCloseCalendar}
                    onChange={onChangeCalendar}
                />
            </GlobalPopup>,
        );
    }

    if (isBusWaterPage) {
        buttons.push(
            <Button2
                className={b('searchButton')}
                key="searchButton"
                onClick={onClickSearchButton}
            >
                <Icon className={b('searchIcon')} glyph={IconGlyph.search} />
                {/* Фэйковый инпут нужен для того, чтобы появилась клавиатура на ios. Иначе поле в саджестах зафокусится, но без клавиатуры и придется тыкать в инпут, для того чтобы она появилась */}
                <input className={b('fakeInput')} />
            </Button2>,
        );

        if (stationFilterIsOpened) {
            otherElements.push(
                <SuggestMobile
                    key="stationSuggests"
                    value={stopText}
                    header={stationKeyset('search-station-header')}
                    placeholder={stationKeyset('search-station-placeholder')}
                    noDataText={stationKeyset('search-station-not-found')}
                    getData={getStationSuggest}
                    onChange={onChangeStopText}
                    onClose={onCloseStationSuggest}
                    onClickElement={onClickStationSuggest}
                    regExpForHighlightingElements={regExpFroFilterSuggest}
                />,
            );
        }
    }

    useEffect(() => {
        if (isPlanePage) {
            reachGoalOnce('time_filter_shown_mobile');
        }

        if (terminalOptions.length > 0) {
            reachGoalOnce('terminal_filter_shown_mobile');
        }
    }, [terminalOptions, isPlanePage]);

    if (!isPlanePage) {
        return (
            <>
                <div className={b({fullWidth: true}, className)}>{buttons}</div>
                {otherElements}
            </>
        );
    }

    buttons.push(
        <Select
            className={b('timeSelect')}
            key="timeSelect"
            options={timeOptions}
            formatTitle={formatTitleTimes}
            value={selectedTimeOption}
            onChange={onChangeTime}
            listItemSize="big"
            type="grayButton"
        />,
    );

    if (terminalOptions.length > 0) {
        buttons.push(
            <Select
                className={b('terminalsSelect')}
                key="terminalsSelect"
                options={terminalOptions}
                value={selectedTerminalOption}
                onChange={onChangeTerminal}
                formatTitle={formatTitleTerminals}
                listItemSize="big"
                type="grayButton"
            />,
        );
    }

    if (buttons.length < 3) {
        return (
            <>
                <div className={b({fullWidth: true}, className)}>{buttons}</div>
                {otherElements}
            </>
        );
    }

    return (
        <>
            <NativeHorizontScroll className={className}>
                <div className={b({scrolling: true})}>
                    <div className={b('filters')}>{buttons}</div>
                </div>
            </NativeHorizontScroll>
            {otherElements}
        </>
    );
}
