import React, {
    memo,
    ReactElement,
    useCallback,
    SyntheticEvent,
    RefObject,
    useEffect,
} from 'react';
import B from 'bem-cn-lite';
import moment from 'moment';

import {ROBOT, HUMAN} from '../../../lib/date/formats';

import DateRobot from '../../../interfaces/date/DateRobot';
import StationTime from '../../../interfaces/state/station/StationTime';
import Platform from '../../../interfaces/Platform';
import DateMoment from '../../../interfaces/date/DateMoment';
import StationDateSpecialValue from '../../../interfaces/date/StationDateSpecialValue';

import useDispatch from '../../useDispatch';
import useSelector from '../../useSelector';
import {useFlags} from '../../../hooks/useFlags';
import timeOptions from '../StationFilters/timeOptions';
import isDateRobot from '../../../lib/date/isDateRobot';
import {stationUrl} from '../../../lib/url/stationUrl';
import scrollWindow from '../../../../client/lib/scrollWindow';
import {reachGoal, reachGoalOnce} from '../../../lib/yaMetrika';

import {changeFilterTime} from '../../../actions/station';

import Link from '../../Link';

import stationKeyset from '../../../i18n/station';

const b = B('StationPlaneNextButton');

const isMobile = process.env.PLATFORM === Platform.mobile;
const timeOptionsWithoutAllTime = timeOptions.filter(
    ({value}) => value !== StationTime.all,
);

interface IStationPlaneNextButton {
    className?: string;
    refToScroll?: RefObject<HTMLElement>;
}

export default memo(StationPlaneNextButton);

function StationPlaneNextButton({
    className,
    refToScroll,
}: IStationPlaneNextButton): ReactElement | null {
    const tld = useSelector(state => state.tld);
    const language = useSelector(state => state.language);
    const flags = useFlags();
    const id = useSelector(state => state.station.id);
    const event = useSelector(state => state.station.event);
    const type = useSelector(state => state.station.type);
    const subtype = useSelector(state => state.station.currentSubtype);
    const mainSubtype = useSelector(state => state.station.mainSubtype);
    const terminalName = useSelector(state => state.station.terminalName);
    const search = useSelector(state => state.station.search);
    const whenDate = useSelector(state => state.station.whenDate);
    const whenSpecial = useSelector(state => state.station.whenSpecial);
    const now = useSelector(state => state.station.now);
    const time = useSelector(state => state.station.time);
    const dispatch = useDispatch();

    const [nextDate, nextSpecial, nextTime] = calculateNextDateOrTime(
        now,
        whenDate,
        time,
    );

    const commonUrlParams = {
        tld,
        language,
        flags,
        id,
        event,
        type,
        subtype,
        mainSubtype,
        terminalName,
        search,
        isMobile,
    };

    const hrefNextDate =
        nextDate || nextSpecial
            ? stationUrl({
                  ...commonUrlParams,
                  date: nextDate,
                  special: nextSpecial,
              })
            : undefined;

    const hrefNextTime = nextTime
        ? stationUrl({
              ...commonUrlParams,
              time: nextTime,
              date: whenDate,
              special: whenSpecial,
          })
        : undefined;

    const onClick = useCallback(
        (e: SyntheticEvent) => {
            reachGoal('station_next_button_click');
            scrollWindow(refToScroll || 0);

            if (nextTime) {
                e.preventDefault();
                dispatch(changeFilterTime(nextTime));
            }
        },
        [nextTime, refToScroll, dispatch],
    );

    const text = getTextForButton(nextDate, nextSpecial, nextTime);

    useEffect(() => {
        if (text) {
            reachGoalOnce('station_next_button_shown');
        }
    }, [text]);

    if (!text) {
        return null;
    }

    return (
        <div className={b({isMobile}, className)}>
            <Link
                href={hrefNextDate || hrefNextTime}
                onClick={onClick}
                className={b('link')}
            >
                {text}
            </Link>
        </div>
    );
}

function calculateNextDateOrTime(
    nowOnStation: DateMoment,

    whenDate?: DateRobot,
    time?: StationTime,
): [
    DateRobot | undefined,
    StationDateSpecialValue | undefined,
    StationTime | undefined,
] {
    if (!time) {
        return [undefined, undefined, undefined];
    }

    const isLastTimePeriod =
        timeOptionsWithoutAllTime[timeOptionsWithoutAllTime.length - 1]
            .value === time;

    const nextDate =
        whenDate && (time === StationTime.all || isLastTimePeriod)
            ? moment(whenDate).add(1, 'd').format(ROBOT)
            : undefined;

    if (nextDate) {
        const specialDate =
            nextDate === moment.parseZone(nowOnStation).format(ROBOT)
                ? StationDateSpecialValue.today
                : nextDate ===
                  moment.parseZone(nowOnStation).add(1, 'd').format(ROBOT)
                ? StationDateSpecialValue.tomorrow
                : undefined;

        return [
            !specialDate && isDateRobot(nextDate) ? nextDate : undefined,
            specialDate,
            undefined,
        ];
    }

    const currentTimeIndex = timeOptionsWithoutAllTime.findIndex(
        ({value}) => value === time,
    );

    if (currentTimeIndex === -1) {
        return [undefined, undefined, undefined];
    }

    return [
        undefined,
        undefined,
        timeOptionsWithoutAllTime[currentTimeIndex + 1]?.value ||
            timeOptionsWithoutAllTime[0].value,
    ];
}

function getTextForButton(
    nextDate?: DateRobot,
    nextSpecial?: StationDateSpecialValue,
    nextTime?: StationTime,
): string {
    if (nextSpecial === StationDateSpecialValue.today) {
        return stationKeyset('schedule-on-today');
    }

    if (nextSpecial === StationDateSpecialValue.tomorrow) {
        return stationKeyset('schedule-on-tomorrow');
    }

    if (nextDate) {
        return stationKeyset('schedule-on-date', {
            date: moment(nextDate).format(HUMAN),
        });
    }

    if (nextTime) {
        const [min, max] = nextTime.split('-');

        return stationKeyset('schedule-on-time', {min, max});
    }

    return '';
}
