import React, {
    useEffect,
    ReactElement,
    SyntheticEvent,
    useContext,
} from 'react';
import B from 'bem-cn-lite';
import upperFirst from 'lodash/upperFirst';
import moment from 'moment';

import ThreadStatus from '../../../interfaces/state/station/ThreadStatus';
import StationEventList from '../../../interfaces/state/station/StationEventList';
import StationDateSpecialValue from '../../../interfaces/date/StationDateSpecialValue';
import IThreadPlane from '../../../interfaces/state/station/IThreadPlane';
import ITerminal from '../../../interfaces/state/station/ITerminal';
import IStationCompany from '../../../interfaces/state/station/IStationCompany';
import IStationThreadStatus from '../../../interfaces/state/station/IStationThreadStatus';
import StationCompaniesById from '../../../interfaces/state/station/StationCompaniesById';
import Tld from '../../../interfaces/Tld';
import Lang from '../../../interfaces/Lang';
import AdditionalThreadStatus from '../../../interfaces/state/station/AdditionalThreadStatus';
import IStationThreadEventDt from '../../../interfaces/state/station/IStationThreadEventDt';
import DateRobot from 'common/interfaces/date/DateRobot';

import useSelector from '../../useSelector';
import getAbsoluteUrlForStaticFromBackend from '../../../lib/url/getAbsoluteUrlForStaticFromBackend';
import getPlaneDirection from '../../../lib/station/getPlaneDirection';
import getCompany from '../../../lib/station/getCompany';
import {CHAR_NBSP} from '../../../lib/stringUtils';
import formatCompanies from './formatCompanies';
import getTimeChangeMessage from './getTimeChangeMessage';
import getMessageFromStatus from './getMessageFromStatus';
import getKeyForPlaneThread from './getKeyForPlaneThread';
import {reachGoalOnce} from '../../../lib/yaMetrika';
import getAviaCompanyLink from '../../../lib/url/getAviaCompanyLink';
import threadTimeIsAlert from './threadTimeIsAlert';
import threadTimeIsNotActual from './threadTimeIsNotActual';
import threadTimeChangeIsAlert from './threadTimeChangeIsAlert';
import getOnThreadRowClick from './getOnThreadRowClick';
import getAviaUrlForStationThread from '../../../lib/station/getAviaUrlForStationThread';
import getCrowdTestingUrl from '../../../lib/url/getCrowdTestingUrl';
import deleteLastSymbol from '../../../lib/string/deleteLastSymbol';
import getDateRobotFromDateMoment from '../../../lib/date/getDateRobotFromDateMoment';

import Link from '../../Link';
import ExceptString from '../../ExceptString/ExceptString';
import StaticContext from '../../../../common/components/StaicContext/StaticContext';

import tableKeyset from '../../../i18n/station-plane-table';

const b = B('StationPlaneTable');

function onClickLink(e: SyntheticEvent<HTMLElement>): void {
    e.preventDefault();
}

interface IGetTableHeadParams {
    showTerminals: boolean;
    showFlightDays: boolean;
    trusted: boolean;
}

function getTableHead({
    showTerminals,
    showFlightDays,
    trusted,
}: IGetTableHeadParams): React.ReactElement {
    return (
        <thead className={b('tableHead')}>
            <tr>
                <th className={b('headTime')}>
                    {tableKeyset('table-heading-time')}
                </th>

                <th className={b('headDirection')}>
                    <div className={b('headDirectionCellValue')}>
                        {tableKeyset('table-heading-direction')}
                    </div>
                </th>

                <th />

                <th className={b('headFlight')}>
                    {tableKeyset('table-heading-flight')}
                </th>

                {showTerminals && (
                    <th
                        className={b('headTerminal', {
                            lastColumn: !showFlightDays && !trusted,
                        })}
                    >
                        {tableKeyset('table-heading-terminal')}
                    </th>
                )}

                {showFlightDays ? (
                    <th className={b('headFlightDays')}>
                        {tableKeyset('table-heading-flight-days')}
                    </th>
                ) : (
                    trusted && (
                        <th className={b('headStatus')}>
                            {tableKeyset('table-heading-status')}
                        </th>
                    )
                )}
            </tr>
        </thead>
    );
}

interface IGetTimeColumnParams {
    eventDt: IStationThreadEventDt;
    whenDate: DateRobot | undefined;
    status: IStationThreadStatus | undefined;
    minutesBetweenEventDtAndActualDt: number | undefined;
}

function getTimeColumn({
    eventDt,
    whenDate,
    status,
    minutesBetweenEventDtAndActualDt,
}: IGetTimeColumnParams): React.ReactElement {
    const expectedDt = eventDt.datetime;
    const actualDt = status?.actualDt;

    const notActual = threadTimeIsNotActual(
        status ?? null,
        minutesBetweenEventDtAndActualDt ?? null,
    );
    const isAlert = threadTimeIsAlert(status ?? null);
    const differentDate =
        expectedDt &&
        actualDt &&
        getDateRobotFromDateMoment(expectedDt) !==
            getDateRobotFromDateMoment(actualDt) &&
        getDateRobotFromDateMoment(expectedDt) !== whenDate;

    const text = differentDate
        ? deleteLastSymbol(
              moment.parseZone(expectedDt).format('HH:mm D MMM'),
              '.',
          )
        : eventDt.time;

    return <td className={b('time', {notActual, isAlert})}>{text}</td>;
}

interface IGetDirectionColumnParams {
    direction: string;

    narrowColumn?: boolean;
}

function getDirectionColumn({
    direction,

    narrowColumn,
}: IGetDirectionColumnParams): React.ReactElement {
    return (
        <td className={b('direction', {narrowColumn})} title={direction}>
            {direction}
        </td>
    );
}

interface IGetCompanyLogoColumnParams {
    isProduction: boolean;
    tld: Tld;
    language: Lang;
    company: IStationCompany | undefined;
}

function getCompanyLogoColumn({
    isProduction,
    tld,
    language,
    company,
}: IGetCompanyLogoColumnParams): React.ReactElement {
    return (
        <td rowSpan={2} className={b('companyLogo')}>
            {company?.icon && (
                <Link
                    href={getAviaCompanyLink({id: company.id, tld, language})}
                >
                    <img
                        className={b('companyLogoImg')}
                        src={getAbsoluteUrlForStaticFromBackend(
                            company.icon,
                            isProduction,
                        )}
                    />
                </Link>
            )}
        </td>
    );
}

interface IGetFlightColumnParams {
    thread: IThreadPlane;
    companiesById: StationCompaniesById;
}

function getFlightColumn({
    thread,
    companiesById,
}: IGetFlightColumnParams): React.ReactElement {
    const {number, companyId, codeshares, isSupplement, aviaLink} = thread;

    const formattedCompanies = formatCompanies(
        companyId,
        number,
        companiesById,
        codeshares,
    );
    const flightNumber = formattedCompanies.map((flight, flightIndex) => {
        return (
            <React.Fragment key={`${flight.companyId}-${flightIndex}`}>
                {`${flightIndex > 0 ? ', ' : ''}${
                    flight.numberLetters
                }${CHAR_NBSP}`}
                <span className={b('flightNumberNumeral')}>
                    {flight.numberNumeral}
                </span>
            </React.Fragment>
        );
    });
    const aviaUrl = aviaLink
        ? getAviaUrlForStationThread(aviaLink, isSupplement)
        : undefined;

    return (
        <td rowSpan={2} className={b('flight')}>
            {aviaUrl ? (
                <Link href={aviaUrl} target="_blank" onClick={onClickLink}>
                    {flightNumber}
                </Link>
            ) : (
                flightNumber
            )}

            <div className={b('companies')}>
                {formattedCompanies
                    .map(flight => flight.company?.title)
                    .filter(Boolean)
                    .join(', ')}
            </div>
        </td>
    );
}

interface IGetTerminalColumnParams {
    isLastColumn: boolean;
    terminalName: string | undefined;
    actualTerminalName: string | undefined;
}

function getTerminalColumn({
    isLastColumn,

    terminalName,
    actualTerminalName,
}: IGetTerminalColumnParams): React.ReactElement {
    return (
        <td
            rowSpan={2}
            className={b('terminal', {
                changed:
                    actualTerminalName && actualTerminalName !== terminalName,
                lastColumn: isLastColumn,
            })}
        >
            {actualTerminalName || terminalName}
        </td>
    );
}

interface IStatusMessageColumnParams {
    showFlightDays: boolean;
    trusted: boolean;
    event: StationEventList;
    daysText: string | undefined;
    status: IStationThreadStatus | undefined;
    terminalName: string | undefined;
    hoursBeforeEvent: number | undefined;
}

function statusMessageColumn({
    showFlightDays,
    trusted,
    event,
    daysText,
    status,
    terminalName,
    hoursBeforeEvent,
}: IStatusMessageColumnParams): React.ReactElement {
    return (
        <td rowSpan={2} className={b('statusMessage')}>
            {showFlightDays && daysText && (
                <ExceptString text={upperFirst(daysText)} />
            )}

            {!showFlightDays &&
                trusted &&
                status &&
                hoursBeforeEvent &&
                getMessageFromStatus(
                    status,
                    event,
                    hoursBeforeEvent,
                    terminalName,
                ).map(message => {
                    return (
                        <div
                            key={message.text}
                            className={b('message', {
                                noRelevantInfo:
                                    message.status === ThreadStatus.unknown,
                                isAlert:
                                    message.status ===
                                    AdditionalThreadStatus.changeTerminal,
                            })}
                        >
                            {message.text}
                        </div>
                    );
                })}
        </td>
    );
}

interface IGetTimeChangeMessageColumnParams {
    eventDt: IStationThreadEventDt;
    event: StationEventList;
    status: IStationThreadStatus | undefined;
    minutesBetweenEventDtAndActualDt: number | undefined;
}

function getTimeChangeMessageColumn({
    eventDt,
    event,
    status,
    minutesBetweenEventDtAndActualDt,
}: IGetTimeChangeMessageColumnParams): React.ReactElement {
    const isAlert = threadTimeChangeIsAlert(
        event,
        status ?? null,
        minutesBetweenEventDtAndActualDt ?? null,
    );

    return (
        <td colSpan={2} className={b('timeChangeMessage', {isAlert})}>
            {status ? getTimeChangeMessage(eventDt, status, event) : ''}
        </td>
    );
}

interface IGetTableParams {
    threads: IThreadPlane[];
    isProduction: boolean;
    companiesById: StationCompaniesById;
    event: StationEventList;
    showTerminals: boolean;
    showFlightDays: boolean;
    trusted: boolean;
    tld: Tld;
    language: Lang;
    isCrowdTesting: boolean;
    whenDate: DateRobot | undefined;
}

function getTableBody({
    threads,
    isProduction,
    companiesById,
    event,
    showTerminals,
    showFlightDays,
    trusted,
    tld,
    language,
    isCrowdTesting,
    whenDate,
}: IGetTableParams): ReactElement[] {
    return threads.map((thread): ReactElement => {
        const {
            companyId,
            daysText,
            routeStations,
            terminalName,
            status,
            hoursBeforeEvent,
            minutesBetweenEventDtAndActualDt,
            eventDt,
            isSupplement,
            aviaLink,
        } = thread;

        const {actualTerminalName} = status || {};

        const company = companyId
            ? getCompany(companyId, companiesById)
            : undefined;
        const isLastColumn = !showFlightDays && !trusted;
        const key = getKeyForPlaneThread(thread);

        const directionNarrowColumn =
            showTerminals || showFlightDays || trusted;
        let aviaUrl =
            aviaLink && getAviaUrlForStationThread(aviaLink, isSupplement);

        if (isCrowdTesting && aviaUrl) {
            aviaUrl = getCrowdTestingUrl(aviaUrl);
        }

        const onClickRow = aviaUrl
            ? getOnThreadRowClick(aviaUrl, false)
            : undefined;

        return (
            <tbody
                key={key}
                className={b('rowContainer', {withLink: Boolean(aviaUrl)})}
                onClick={onClickRow}
            >
                <tr className={b('threadRow')}>
                    {getTimeColumn({
                        eventDt,
                        whenDate,
                        status,
                        minutesBetweenEventDtAndActualDt,
                    })}

                    {getDirectionColumn({
                        direction: getPlaneDirection(routeStations, event),
                        narrowColumn: directionNarrowColumn,
                    })}

                    {getCompanyLogoColumn({
                        isProduction,
                        tld,
                        language,
                        company,
                    })}

                    {getFlightColumn({thread, companiesById})}

                    {showTerminals &&
                        getTerminalColumn({
                            isLastColumn,
                            terminalName,
                            actualTerminalName,
                        })}

                    {(showFlightDays || trusted) &&
                        statusMessageColumn({
                            showFlightDays,
                            trusted,
                            event,
                            daysText,
                            status,
                            terminalName,
                            hoursBeforeEvent,
                        })}
                </tr>

                <tr>
                    {getTimeChangeMessageColumn({
                        eventDt,
                        event,
                        status,
                        minutesBetweenEventDtAndActualDt,
                    })}
                </tr>
            </tbody>
        );
    });
}

interface IStationPlaneTableProps {
    threads: IThreadPlane[];
    companiesById: StationCompaniesById;
    terminals: ITerminal[];
    trusted: boolean;
    event: StationEventList;
    whenSpecial: StationDateSpecialValue | undefined;

    className?: string;
}

function StationPlaneTable({
    threads,
    companiesById,
    terminals,
    trusted,
    whenSpecial,
    event,

    className,
}: IStationPlaneTableProps): React.ReactElement {
    const isProduction = useSelector(state => state.environment.production);
    const tld = useSelector(state => state.tld);
    const language = useSelector(state => state.language);
    const whenDate = useSelector(state => state.station.whenDate);

    const {isCrowdTesting} = useContext(StaticContext);

    const showTerminals = Boolean(terminals.length);
    const showFlightDays = whenSpecial === StationDateSpecialValue.allDays;

    useEffect(() => {
        if (showFlightDays) {
            reachGoalOnce('flight_days_shown_desktop');
        } else if (trusted) {
            reachGoalOnce('status_column_shown_desktop');
        }
    }, [showFlightDays, trusted]);

    return (
        <table className={b(undefined, className)}>
            {getTableHead({showTerminals, showFlightDays, trusted})}

            {getTableBody({
                threads,
                isProduction,
                companiesById,
                event,
                showTerminals,
                showFlightDays,
                trusted,
                tld,
                language,
                isCrowdTesting,
                whenDate,
            })}
        </table>
    );
}

export default React.memo(StationPlaneTable);
