import React, {SyntheticEvent, useContext} from 'react';
import B from 'bem-cn-lite';
import moment from 'moment';

import ThreadStatus from '../../../interfaces/state/station/ThreadStatus';
import StationEventList from '../../../interfaces/state/station/StationEventList';
import ITerminal from '../../../interfaces/state/station/ITerminal';
import IStationCompany from '../../../interfaces/state/station/IStationCompany';
import StationDateSpecialValue from '../../../interfaces/date/StationDateSpecialValue';
import StationCompaniesById from '../../../interfaces/state/station/StationCompaniesById';
import IThreadPlane from '../../../interfaces/state/station/IThreadPlane';
import Tld from '../../../interfaces/Tld';
import Lang from '../../../interfaces/Lang';
import AdditionalThreadStatus from '../../../interfaces/state/station/AdditionalThreadStatus';
import DateRobot from '../../../interfaces/date/DateRobot';
import StationType from '../../../interfaces/state/station/StationType';
import IStationThreadStatus from '../../../interfaces/state/station/IStationThreadStatus';
import IconGlyph from '../../../interfaces/components/IconGlyph';
import IStationThreadEventDt from '../../../interfaces/state/station/IStationThreadEventDt';

import useSelector from '../../useSelector';
import getAbsoluteUrlForStaticFromBackend from '../../../lib/url/getAbsoluteUrlForStaticFromBackend';
import getCompany from '../../../lib/station/getCompany';
import getPlaneDirection from '../../../lib/station/getPlaneDirection';
import {CHAR_NBSP} from '../../../lib/stringUtils';
import getTimeChangeMessage from './getTimeChangeMessage';
import getMessageFromStatus from './getMessageFromStatus';
import formatCompanies from './formatCompanies';
import getKeyForPlaneThread from './getKeyForPlaneThread';
import {reachGoal} from '../../../lib/yaMetrika';
import getAviaCompanyLink from '../../../lib/url/getAviaCompanyLink';
import getTimeFromDateMoment from '../../../lib/date/getTimeFromDateMoment';
import threadTimeIsNotActual from './threadTimeIsNotActual';
import threadTimeIsAlert from './threadTimeIsAlert';
import threadTimeChangeIsAlert from './threadTimeChangeIsAlert';
import getOnThreadRowClick from './getOnThreadRowClick';
import getWhenDateText from '../getWhenDateText';
import getAviaUrlForStationThread from '../../../lib/station/getAviaUrlForStationThread';
import getCrowdTestingUrl from '../../../lib/url/getCrowdTestingUrl';
import getDateRobotFromDateMoment from '../../../lib/date/getDateRobotFromDateMoment';
import deleteLastSymbol from '../../../lib/string/deleteLastSymbol';

import Icon from '../../../components/Icon/Icon';
import Link from '../../Link';
import StationNoThreadsText from '../StationTable/StationNoThreadsText';
import ExceptString from '../../ExceptString/ExceptString';
import StaticContext from '../../../../common/components/StaicContext/StaticContext';

import tableKeyset from '../../../i18n/station-plane-table';
import stationKeyset from '../../../i18n/station';

const b = B('StationPlaneTable');

function onClickLink(e: SyntheticEvent<HTMLElement>): void {
    e.preventDefault();
}

function getTableHead(showTerminals: boolean): React.ReactElement {
    return (
        <thead className={b('tableHead')}>
            <tr>
                <th className={b('headTime')}>
                    {tableKeyset('table-heading-time')}
                </th>

                <th />

                <th className={b('headFlight')}>
                    {tableKeyset('table-heading-flight')}
                </th>

                {showTerminals && (
                    <th className={b('headTerminal')} colSpan={2}>
                        {tableKeyset('table-heading-terminal')}
                    </th>
                )}
            </tr>
        </thead>
    );
}

interface IGetTimeColumnParams {
    event: StationEventList;
    eventDt: IStationThreadEventDt;
    status: IStationThreadStatus | undefined;
    minutesBetweenEventDtAndActualDt: number | undefined;
    whenDate: DateRobot | undefined;
}

function getTimeColumn({
    event,
    eventDt,

    status,
    minutesBetweenEventDtAndActualDt,
    whenDate,
}: IGetTimeColumnParams): React.ReactElement {
    const actualDt = status?.actualDt;
    const expectedDt = eventDt.datetime;
    const expectedTime = eventDt.time;

    const notActual = threadTimeIsNotActual(
        status || null,
        minutesBetweenEventDtAndActualDt || null,
    );
    const isAlert = threadTimeIsAlert(status || null);
    const differentDate =
        actualDt &&
        expectedDt &&
        getDateRobotFromDateMoment(actualDt) !==
            getDateRobotFromDateMoment(expectedDt) &&
        getDateRobotFromDateMoment(expectedDt) !== whenDate;
    const expectedDate = differentDate
        ? deleteLastSymbol(moment.parseZone(expectedDt).format('D MMM'), '.')
        : undefined;

    const actualTime = actualDt && getTimeFromDateMoment(actualDt);
    const actualDate = differentDate
        ? deleteLastSymbol(moment.parseZone(actualDt).format('D MMM'), '.')
        : undefined;
    const actualTimeIsAlert = threadTimeChangeIsAlert(
        event,
        status || null,
        minutesBetweenEventDtAndActualDt || null,
    );

    return (
        <td className={b('timeColumn')}>
            <div className={b('time')}>
                <span className={b('expectedTime', {notActual, isAlert})}>
                    {expectedTime}
                </span>

                {expectedDate && (
                    <span className={b('expectedDate', {notActual, isAlert})}>
                        {expectedDate}
                    </span>
                )}

                {notActual && actualTime && (
                    <span
                        className={b('actualTime', {
                            isAlert: actualTimeIsAlert,
                        })}
                    >
                        {actualTime}
                    </span>
                )}

                {notActual && actualDate && (
                    <span
                        className={b('actualDate', {
                            isAlert: actualTimeIsAlert,
                        })}
                    >
                        {actualDate}
                    </span>
                )}
            </div>
        </td>
    );
}

function companyOnClick(): void {
    reachGoal('flight_company_click_mobile');
}

interface IGetCompanyLogoColumnParams {
    isProduction: boolean;
    tld: Tld;
    language: Lang;
    company: IStationCompany | undefined;
}

function getCompanyLogoColumn({
    isProduction,
    tld,
    language,
    company,
}: IGetCompanyLogoColumnParams): React.ReactElement {
    return (
        <td className={b('companyLogo')}>
            {company?.icon && (
                <Link
                    href={getAviaCompanyLink({id: company.id, tld, language})}
                    onClick={companyOnClick}
                >
                    <img
                        className={b('companyLogoImg')}
                        src={getAbsoluteUrlForStaticFromBackend(
                            company.icon,
                            isProduction,
                        )}
                    />
                </Link>
            )}
        </td>
    );
}

interface IGetFlightColumnParams {
    thread: IThreadPlane;
    companiesById: StationCompaniesById;
    showFlightDays: boolean;
    event: StationEventList;
    isCrowdTesting: boolean;
    statusValue: ThreadStatus | undefined;
}

function getFlightColumn({
    thread,
    companiesById,
    showFlightDays,
    event,
    isCrowdTesting,
    statusValue,
}: IGetFlightColumnParams): React.ReactElement {
    const {
        number,
        eventDt,
        companyId,
        daysText,
        routeStations,
        terminalName,
        codeshares,
        status,
        aviaLink,
        isSupplement,
        hoursBeforeEvent,
    } = thread;

    const formattedCompanies = formatCompanies(
        companyId,
        number,
        companiesById,
        codeshares,
    );
    let aviaUrl =
        aviaLink && getAviaUrlForStationThread(aviaLink, isSupplement);

    if (isCrowdTesting && aviaUrl) {
        aviaUrl = getCrowdTestingUrl(aviaUrl);
    }

    const flightNumbers = formattedCompanies
        .map(flight => flight.number.replace(/\s+/g, CHAR_NBSP))
        .join(', ');

    return (
        <td className={b('flightColumn')} colSpan={2}>
            <div
                className={b('direction', {
                    cancelled: statusValue === ThreadStatus.cancelled,
                })}
            >
                {getPlaneDirection(routeStations, event)}
            </div>

            <div>
                {aviaUrl ? (
                    <Link href={aviaLink} target="_blank" onClick={onClickLink}>
                        {flightNumbers}
                    </Link>
                ) : (
                    flightNumbers
                )}

                <span className={b('companies')}>
                    {formattedCompanies
                        .map(flight => flight.company?.title)
                        .filter(Boolean)
                        .join(', ')}
                </span>
            </div>

            {showFlightDays ? (
                daysText && <ExceptString text={daysText} />
            ) : (
                <React.Fragment>
                    <span className={b('timeChangeMessage')}>
                        {status
                            ? getTimeChangeMessage(eventDt, status, event, true)
                            : ''}
                    </span>

                    {status && hoursBeforeEvent
                        ? getMessageFromStatus(
                              status,
                              event,
                              hoursBeforeEvent,
                              terminalName,
                          ).map(message => (
                              <div
                                  key={message.text}
                                  className={b('message', {
                                      isAlert:
                                          message.status ===
                                          AdditionalThreadStatus.changeTerminal,
                                  })}
                              >
                                  {message.text}
                              </div>
                          ))
                        : ''}
                </React.Fragment>
            )}
        </td>
    );
}

interface IGetTerminalColumnParams {
    terminalName: string | undefined;
    actualTerminalName: string | undefined;
}

function getTerminalColumn({
    terminalName,
    actualTerminalName,
}: IGetTerminalColumnParams): React.ReactElement {
    return (
        <td
            className={b('terminalColumn', {
                changed:
                    actualTerminalName && actualTerminalName !== terminalName,
            })}
        >
            {actualTerminalName || terminalName}
        </td>
    );
}

interface IGetTableBodyParams {
    threads: IThreadPlane[];
    companiesById: StationCompaniesById;
    event: StationEventList;
    showTerminals: boolean;
    showFlightDays: boolean;
    isProduction: boolean;
    tld: Tld;
    language: Lang;
    isCrowdTesting: boolean;
    whenDate: DateRobot | undefined;
}

function getTableBody({
    threads,
    companiesById,
    event,
    showTerminals,
    showFlightDays,
    isProduction,
    tld,
    language,
    isCrowdTesting,

    whenDate,
}: IGetTableBodyParams): React.ReactElement {
    const result: JSX.Element[] = [];

    threads.forEach((thread): void => {
        const {
            companyId,
            terminalName,
            status,
            minutesBetweenEventDtAndActualDt,
            eventDt,
            aviaLink,
            isSupplement,
        } = thread;

        const {status: statusValue, actualTerminalName} = status || {};

        const company = getCompany(companyId, companiesById);
        let aviaUrl =
            aviaLink && getAviaUrlForStationThread(aviaLink, isSupplement);

        if (isCrowdTesting && aviaUrl) {
            aviaUrl = getCrowdTestingUrl(aviaUrl);
        }

        const onClickRow = aviaUrl
            ? getOnThreadRowClick(aviaUrl, true)
            : undefined;

        result.push(
            <tr
                key={getKeyForPlaneThread(thread)}
                className={b('tableRow')}
                onClick={onClickRow}
            >
                {getTimeColumn({
                    event,
                    eventDt,
                    status,
                    minutesBetweenEventDtAndActualDt,
                    whenDate,
                })}

                {getCompanyLogoColumn({isProduction, tld, language, company})}

                {getFlightColumn({
                    thread,
                    companiesById,
                    showFlightDays,
                    event,
                    isCrowdTesting,
                    statusValue,
                })}

                {showTerminals &&
                    getTerminalColumn({terminalName, actualTerminalName})}
            </tr>,
        );
    });

    return <tbody>{result}</tbody>;
}

interface IStationPlaneTableProps {
    threads: IThreadPlane[];
    companiesById: StationCompaniesById;
    event: StationEventList;
    terminals: ITerminal[];
    type: StationType;
    whenSpecial: StationDateSpecialValue | undefined;
    whenDate: DateRobot | undefined;

    className?: string;
}

function StationPlaneTable({
    threads,
    companiesById,
    event,
    terminals,
    type,
    whenSpecial,
    whenDate,

    className,
}: IStationPlaneTableProps): React.ReactElement {
    const isProduction = useSelector(state => state.environment.production);
    const tld = useSelector(state => state.tld);
    const language = useSelector(state => state.language);

    const {isCrowdTesting} = useContext(StaticContext);

    const showTerminals = Boolean(terminals.length);
    const showFlightDays = whenSpecial === StationDateSpecialValue.allDays;

    if (threads.length === 0) {
        return (
            <StationNoThreadsText
                className={b({noThreads: true}, className)}
                isMobile
            />
        );
    }

    const whenDateText = getWhenDateText(type, event, whenDate, whenSpecial);

    return (
        <div>
            {whenDateText && (
                <div className={b('whenDate')}>{whenDateText}</div>
            )}
            <div
                className={b('localTime', {
                    onAllDays: whenSpecial === StationDateSpecialValue.allDays,
                })}
            >
                <Icon className={b('localTimeIcon')} glyph={IconGlyph.alert2} />
                {stationKeyset('local-time-disclaimer')}
            </div>
            <table className={b(undefined, className)}>
                {getTableHead(showTerminals)}

                {getTableBody({
                    threads,
                    companiesById,
                    event,
                    showTerminals,
                    showFlightDays,
                    isProduction,
                    tld,
                    language,
                    isCrowdTesting,
                    whenDate,
                })}
            </table>
        </div>
    );
}

export default React.memo(StationPlaneTable);
