import {CHAR_EM_DASH, CHAR_MIDDLE_DASH} from '../../../lib/stringUtils';

import ThreadStatus from '../../../interfaces/state/station/ThreadStatus';
import AdditionalThreadStatus from '../../../interfaces/state/station/AdditionalThreadStatus';
import StationEventList from '../../../interfaces/state/station/StationEventList';
import IStationThreadStatus from '../../../interfaces/state/station/IStationThreadStatus';

import tableKeyset from '../../../i18n/station-plane-table';

const regexpForDashes = new RegExp(`[\\-${CHAR_EM_DASH}]`, 'g');

interface IMessage {
    text: string;

    status?: ThreadStatus | AdditionalThreadStatus;
}

export default function getMessageFromStatus(
    statusObj: IStationThreadStatus,
    event: StationEventList,
    hoursBeforeEvent: number,

    terminal?: string,
    forInformer?: boolean,
): IMessage[] {
    const messages: IMessage[] = [];

    if (hoursBeforeEvent > 24) {
        return messages;
    }

    const {status, actualTerminalName, checkInDesks, gate, baggageCarousels} =
        statusObj;

    if (status === ThreadStatus.unknown) {
        return [{text: tableKeyset('unknown'), status: ThreadStatus.unknown}];
    }

    if (status === ThreadStatus.cancelled) {
        if (forInformer) {
            return [
                {text: tableKeyset('canceled'), status: ThreadStatus.cancelled},
            ];
        }

        return [];
    }

    if (terminal && forInformer) {
        return [
            {
                text: tableKeyset('terminal-name', {
                    name: statusObj.actualTerminalName || terminal,
                }),
                status:
                    statusObj.actualTerminalName &&
                    statusObj.actualTerminalName !== terminal
                        ? AdditionalThreadStatus.changeTerminal
                        : undefined,
            },
        ];
    }

    if (terminal && actualTerminalName && terminal !== actualTerminalName) {
        if (!forInformer) {
            messages.push({
                text: tableKeyset('change-terminal'),
                status: AdditionalThreadStatus.changeTerminal,
            });
        }
    }

    if (event === StationEventList.departure) {
        if (status === ThreadStatus.onTime) {
            messages.push({text: tableKeyset('departure-on-time')});
        }

        if (status === ThreadStatus.early) {
            messages.push({text: tableKeyset('early')});
        }

        if (status === ThreadStatus.departed) {
            messages.push({text: tableKeyset('departed')});
        }

        if (!forInformer) {
            if (checkInDesks && status !== ThreadStatus.departed) {
                const numbersFormatted = checkInDesks.replace(
                    regexpForDashes,
                    CHAR_MIDDLE_DASH,
                );

                messages.push({
                    text: tableKeyset('check-in-desks', {
                        numbers: numbersFormatted,
                    }),
                });
            }

            if (gate && status !== ThreadStatus.departed) {
                messages.push({
                    text: tableKeyset('departure-gate', {number: gate}),
                });
            }
        }

        if (forInformer && status === ThreadStatus.delayed) {
            messages.push({
                text: tableKeyset('delayed'),
                status: ThreadStatus.delayed,
            });
        }

        return messages;
    }

    if (event === StationEventList.arrival) {
        if (status === ThreadStatus.early) {
            messages.push({text: tableKeyset('arrival-early')});
        }

        if (status === ThreadStatus.onTime) {
            messages.push({text: tableKeyset('arrival-on-time')});
        }

        if (status === ThreadStatus.delayed) {
            messages.push({text: tableKeyset('arrival-delayed')});
        }

        if (status === ThreadStatus.arrived) {
            messages.push({text: tableKeyset('arrived')});
        }

        if (!forInformer) {
            if (gate) {
                messages.push({
                    text: tableKeyset('arrival-gate', {number: gate}),
                });
            }

            if (baggageCarousels) {
                messages.push({
                    text: tableKeyset('baggage-claim-carousels', {
                        number: baggageCarousels,
                    }),
                });
            }
        }
    }

    return messages;
}
