import moment from 'moment';

import ThreadStatus from '../../../interfaces/state/station/ThreadStatus';
import StationEventList from '../../../interfaces/state/station/StationEventList';
import IStationThreadStatus from '../../../interfaces/state/station/IStationThreadStatus';
import IStationThreadEventDt from '../../../interfaces/state/station/IStationThreadEventDt';

import getTimeFromDateMoment from '../../../lib/date/getTimeFromDateMoment';
import getDateRobotFromDateMoment from '../../../lib/date/getDateRobotFromDateMoment';
import deleteLastSymbol from '../../../lib/string/deleteLastSymbol';

import tableKeyset from '../../../i18n/station-plane-table';

export default function getTimeChangeMessage(
    eventDt: IStationThreadEventDt,
    statusObj: IStationThreadStatus,
    event: StationEventList,

    isMobile?: boolean,
): string {
    const {status, diverted, actualDt} = statusObj;
    const expectedDt = eventDt.datetime;

    if (status === ThreadStatus.cancelled) {
        return tableKeyset('canceled');
    }

    if (diverted) {
        return tableKeyset('diverted', {title: diverted.title});
    }

    if (isMobile) {
        return '';
    }

    const differentDate =
        expectedDt &&
        actualDt &&
        getDateRobotFromDateMoment(expectedDt) !==
            getDateRobotFromDateMoment(actualDt);
    const formattedActualTime = differentDate
        ? deleteLastSymbol(
              moment.parseZone(actualDt).format('HH:mm D MMM'),
              '.',
          )
        : actualDt && getTimeFromDateMoment(actualDt);

    if (event === StationEventList.arrival) {
        if (status === ThreadStatus.delayed && formattedActualTime) {
            return tableKeyset('arrival-delayed-in-time', {
                time: formattedActualTime,
            });
        }

        if (status === ThreadStatus.early && formattedActualTime) {
            return tableKeyset('arrival-early-in-time', {
                time: formattedActualTime,
            });
        }

        if (status === ThreadStatus.arrived) {
            return formattedActualTime
                ? tableKeyset('arrived-in-time', {time: formattedActualTime})
                : tableKeyset('arrived');
        }
    }

    if (event === StationEventList.departure) {
        if (status === ThreadStatus.delayed) {
            return formattedActualTime
                ? tableKeyset('departure-delayed', {time: formattedActualTime})
                : tableKeyset('delayed');
        }

        if (status === ThreadStatus.early) {
            return formattedActualTime
                ? tableKeyset('early-in-time', {time: formattedActualTime})
                : tableKeyset('early');
        }
    }

    return '';
}
