import React, {ReactElement, memo, Fragment} from 'react';
import moment from 'moment';

import {HUMAN} from '../../../lib/date/formats';

import StationDateSpecialValue from '../../../interfaces/date/StationDateSpecialValue';
import StationSubtype from '../../../interfaces/state/station/StationSubtype';
import StationEventList from '../../../interfaces/state/station/StationEventList';
import StationType from '../../../interfaces/state/station/StationType';

import useSelector from '../../useSelector';
import {useFlags} from '../../../hooks/useFlags';
import {stationUrl} from '../../../lib/url/stationUrl';

import Link from '../../Link';

import stationKeyset from '../../../i18n/station';

type OtherVariant = {text: string; href: string};

interface IGetNoThreadsText {
    isMobile: boolean;

    className?: string;
}

export default memo(StationNoThreadsText);

function StationNoThreadsText({
    isMobile,

    className,
}: IGetNoThreadsText): ReactElement {
    const tld = useSelector(state => state.tld);
    const language = useSelector(state => state.language);
    const id = useSelector(state => state.station.id);
    const type = useSelector(state => state.station.type);
    const subtypes = useSelector(state => state.station.subtypes);
    const event = useSelector(state => state.station.event);
    const mainSubtype = useSelector(state => state.station.mainSubtype);
    const whenDate = useSelector(state => state.station.whenDate);
    const whenSpecial = useSelector(state => state.station.whenSpecial);
    const currentSubtype = useSelector(state => state.station.currentSubtype);
    const directionCode = useSelector(state => state.station.directionCode);
    const flags = useFlags();

    let noThreadText = '';

    switch (true) {
        case currentSubtype === StationSubtype.train &&
            event === StationEventList.departure &&
            whenDate:
            noThreadText = stationKeyset('no-threads-train-departure-on-date', {
                date: moment(whenDate).format(HUMAN),
            });
            break;
        case currentSubtype === StationSubtype.train &&
            event === StationEventList.departure &&
            whenSpecial === StationDateSpecialValue.allDays:
            noThreadText = stationKeyset(
                'no-threads-train-departure-on-all-days',
            );
            break;
        case currentSubtype === StationSubtype.train &&
            event === StationEventList.arrival &&
            whenDate:
            noThreadText = stationKeyset('no-threads-train-arrival-on-date', {
                date: moment(whenDate).format(HUMAN),
            });
            break;
        case currentSubtype === StationSubtype.train &&
            event === StationEventList.arrival &&
            whenSpecial === StationDateSpecialValue.allDays:
            noThreadText = stationKeyset(
                'no-threads-train-arrival-on-all-days',
            );
            break;
        case currentSubtype === StationSubtype.suburban &&
            event === StationEventList.departure &&
            whenDate:
            noThreadText = stationKeyset(
                'no-threads-suburban-departure-on-date',
                {date: moment(whenDate).format(HUMAN)},
            );
            break;
        case currentSubtype === StationSubtype.suburban &&
            event === StationEventList.departure &&
            whenSpecial === StationDateSpecialValue.allDays:
            noThreadText = stationKeyset(
                'no-threads-suburban-departure-on-all-days',
            );
            break;
        case currentSubtype === StationSubtype.suburban &&
            event === StationEventList.arrival &&
            whenDate:
            noThreadText = stationKeyset(
                'no-threads-suburban-arrival-on-date',
                {date: moment(whenDate).format(HUMAN)},
            );
            break;
        case currentSubtype === StationSubtype.suburban &&
            event === StationEventList.arrival &&
            whenSpecial === StationDateSpecialValue.allDays:
            noThreadText = stationKeyset(
                'no-threads-suburban-arrival-on-all-days',
            );
            break;
        case type === StationType.bus || type === StationType.water:
            noThreadText = stationKeyset('no-threads-bus');
            break;

        default:
            noThreadText = stationKeyset('no-threads-default');
    }

    const otherViewVariants: OtherVariant[] = [];
    const commonUrlParameters = {
        id,
        tld,
        language,
        flags,
        type,
        mainSubtype: mainSubtype ?? undefined,
        isMobile,
        event,
        direction: directionCode ?? undefined,
    };

    if (whenDate) {
        const href = stationUrl({
            ...commonUrlParameters,
            subtype: currentSubtype ?? undefined,
            special: StationDateSpecialValue.allDays,
        });

        otherViewVariants.push({
            text: stationKeyset('schedule-on-all-days'),
            href,
        });
    }

    subtypes
        .filter(subtype => subtype !== currentSubtype)
        .forEach(subtype => {
            if (
                subtype === StationSubtype.train ||
                subtype === StationSubtype.suburban
            ) {
                const href = stationUrl({
                    ...commonUrlParameters,
                    subtype,
                    special: whenSpecial ?? undefined,
                    date: whenDate,
                });
                const text =
                    subtype === StationSubtype.train
                        ? stationKeyset('schedule-train')
                        : stationKeyset('schedule-suburban');

                otherViewVariants.push({
                    text,
                    href,
                });
            }
        });

    return (
        <div className={className}>
            {noThreadText}

            {otherViewVariants.length !== 0 &&
                ` ${stationKeyset('try-view-schedule')}`}

            {otherViewVariants.map(({text, href}, index) => (
                <Fragment key={`${text}-${href}`}>
                    <Link href={href}>{text}</Link>

                    {index !== otherViewVariants.length - 1 ? ', ' : null}
                </Fragment>
            ))}

            {otherViewVariants.length !== 0 && '.'}
        </div>
    );
}
