import React, {ReactNode} from 'react';
import B from 'bem-cn-lite';
import moment from 'moment';

import {ROBOT, HUMAN_SHORT} from '../../../lib/date/formats';

import StationSubtype from '../../../interfaces/state/station/StationSubtype';
import Dispatch from '../../../interfaces/actions/Dispatch';
import IThreadRailroad from '../../../interfaces/state/station/IThreadRailroad';
import DateRobot from '../../../interfaces/date/DateRobot';
import StationDateSpecialValue from '../../../interfaces/date/StationDateSpecialValue';
import Tld from '../../../interfaces/Tld';
import Lang from '../../../interfaces/Lang';
import IStops from '../../../interfaces/state/station/IStops';
import StopsType from '../../../interfaces/state/station/StopsType';
import StationEventList from '../../../interfaces/state/station/StationEventList';
import {TransportType} from '../../../lib/transportType';

import useDispatch from '../../useDispatch';
import useSelector from '../../useSelector';

import getShowGoneThreadsButtonText from './getShowGoneThreadsButtonText';
import {makeCacheable} from '../../../lib/cache';
import getThreadUrl from '../../../lib/url/getThreadUrl';
import getSubtitle from '../../../lib/station/getSubtitle';

import {setGoneThreadsAreOpened} from '../../../actions/station';

import Link from '../../Link';
import Arrow from '../../Arrow/Arrow';
import TransportIcon from '../../TransportIcon/TransportIcon';
import ExceptString from '../../ExceptString/ExceptString';
import StationRunDays from '../../StationRunDays/StationRunDays';

import stationKeyset from '../../../i18n/station';

const b = B('StationTable');

const getOnGoneThreadsButtonClick = makeCacheable(
    (dispatch: Dispatch, goneThreadsAreOpened: boolean) => (): void => {
        return dispatch(setGoneThreadsAreOpened(!goneThreadsAreOpened));
    },
);

function getShowGoneThreadsButton(
    goneRows: React.ReactElement[],
    goneThreadsAreOpened: boolean,
    event: StationEventList,
    dispatch: Dispatch,
    currentSubtype?: StationSubtype,
): React.ReactElement {
    return (
        <tr className={b('showGoneThreads')} key="showGone">
            <td colSpan={4}>
                <span
                    className={b('showGoneThreadsButton', {
                        isOpened: goneThreadsAreOpened,
                    })}
                    onClick={getOnGoneThreadsButtonClick(
                        dispatch,
                        goneThreadsAreOpened,
                    )}
                >
                    <span className={b('buttonText')}>
                        {getShowGoneThreadsButtonText(
                            goneThreadsAreOpened,
                            event,
                            goneRows.length,
                            currentSubtype ?? null,
                        )}
                    </span>
                    <Arrow
                        className={b('goneButtonIcon')}
                        direction={goneThreadsAreOpened ? 'up' : 'down'}
                    />
                </span>
            </td>
        </tr>
    );
}

function getStops(
    stops?: IStops,
    whenSpecial?: StationDateSpecialValue,
): React.ReactElement | null {
    if (!stops) {
        return null;
    }

    const {type: stopType, text: stopText} = stops;

    let type = '';
    let text: ReactNode;

    switch (stopType) {
        case StopsType.everywhere:
            type = stationKeyset('stops-everywhere');
            break;
        case StopsType.nonstop:
            type = stationKeyset('stops-nonstop');
            break;
        case StopsType.stops:
            type = stopText
                ? stationKeyset('stops-stops')
                : stationKeyset('stops-nonstop');
            text = stopText ? <ExceptString text={stopText} /> : undefined;
    }

    const onAllDays = whenSpecial === StationDateSpecialValue.allDays;

    return (
        <div className={b('stops')}>
            <span className={b('stopsType', {onAllDays})}>{type}</span> {text}
        </div>
    );
}

function getDaysText(
    daysText?: string,
    runDaysText?: string,
    exceptDaysText?: string,
    transportType?: TransportType,
): React.ReactElement | null {
    const onTrain = transportType === TransportType.train;

    const content = (
        <StationRunDays
            daysText={daysText ?? null}
            runDaysText={runDaysText ?? null}
            exceptDaysText={exceptDaysText ?? null}
            fontWeightForExcept="normal"
        />
    );

    if (!content) {
        return null;
    }

    return <div className={b('schedule', {onTrain})}>{content}</div>;
}

function getTableBodies(
    threads: IThreadRailroad[],
    id: number,
    goneThreadsAreOpened: boolean,
    dispatch: Dispatch,
    event: StationEventList,
    tld: Tld,
    language: Lang,
    now: number,
    whenDate?: DateRobot,
    whenSpecial?: StationDateSpecialValue,
    currentSubtype?: StationSubtype,
): React.ReactElement[] {
    const rows: React.ReactElement[] = [];
    const goneRows: React.ReactElement[] = [];
    let addingGoneThreads = whenSpecial !== StationDateSpecialValue.allDays;
    const nowMoment = moment(now);
    let addedShowGoneThreads = false;

    threads.forEach(thread => {
        const {
            title,
            number,
            transportSubtype,
            deluxeTrainTitle,
            isAeroExpress,
            transportType,
            platform,
            stops,
            isExpress,
            daysText,
            canonicalUid,
            departure,
            departureFrom,
            eventDt: {time, datetime},
            runDaysText,
            exceptDaysText,
        } = thread;

        const subtitle = getSubtitle(
            transportType,
            isAeroExpress,
            transportSubtype,
            deluxeTrainTitle,
        );

        const dateMoment = datetime ? moment.parseZone(datetime) : null;
        const threadIsGone = dateMoment && dateMoment.isBefore(nowMoment);

        if (addingGoneThreads && !threadIsGone) {
            addingGoneThreads = false;

            if (goneRows.length) {
                rows.push(
                    getShowGoneThreadsButton(
                        goneRows,
                        goneThreadsAreOpened,
                        event,
                        dispatch,
                        currentSubtype,
                    ),
                );
                addedShowGoneThreads = true;
            }
        }

        const target = threadIsGone ? goneRows : rows;

        const onAnotherDay = Boolean(
            dateMoment && dateMoment.format(ROBOT) !== whenDate,
        );
        const shouldAddDate = onAnotherDay;

        const threadUrl = getThreadUrl({
            tld,
            language,
            canonicalUid,
            departure,
            stationFromId: departureFrom ? id : undefined,
            departureFromMoment: departureFrom
                ? moment.parseZone(departureFrom)
                : undefined,
        });

        target.push(
            <tr className={b('tableRow')} key={threadUrl}>
                <td className={b('threadTimeCell')}>
                    <div className={b('wrapper')}>
                        <div className={b('innerWrapper')}>
                            <div className={b('threadTime')}>{time}</div>
                            {dateMoment && shouldAddDate && (
                                <div
                                    className={b('threadDate', {
                                        displayedOnPrint: onAnotherDay,
                                    })}
                                >
                                    {dateMoment
                                        .format(HUMAN_SHORT)
                                        .toLowerCase()}
                                </div>
                            )}
                        </div>
                    </div>
                </td>
                <td className={b('logoBlock')}>
                    <div className={b('wrapper')}>
                        <div className={b('innerWrapper')}>
                            <TransportIcon
                                className={b('transportLogo')}
                                transportType={transportType}
                                isExpress={isExpress}
                                isAeroExpress={isAeroExpress}
                            />
                        </div>
                    </div>
                </td>
                <td className={b('titleCell')}>
                    <div className={b('wrapper')}>
                        <div className={b('innerWrapper')}>
                            <div className={b('titleBlock')}>
                                <div className={b('threadName')}>
                                    <Link
                                        className={b('threadTitle')}
                                        href={threadUrl}
                                    >
                                        {title}
                                    </Link>
                                    <span className={b('threadNumber')}>
                                        {number}
                                    </span>
                                </div>
                                {subtitle && (
                                    <div className={b('threadSubtitle')}>
                                        {subtitle}
                                    </div>
                                )}
                                {platform && (
                                    <div className={b('threadPlatform')}>
                                        {platform}
                                    </div>
                                )}
                            </div>
                        </div>
                    </div>
                </td>
                <td className={b('threadInfo')}>
                    <div className={b('wrapper')}>
                        <div className={b('innerWrapper')}>
                            {getDaysText(
                                daysText,
                                runDaysText,
                                exceptDaysText,
                                transportType,
                            )}

                            {getStops(stops, whenSpecial)}
                        </div>
                    </div>
                </td>
            </tr>,
        );
    });

    const result: React.ReactElement[] = [];

    if (goneRows && goneRows.length) {
        result.push(
            <tbody
                key="gone"
                className={b('goneRows', {
                    isOpened: !addedShowGoneThreads || goneThreadsAreOpened,
                })}
            >
                {goneRows}
            </tbody>,
        );
    }

    if (rows.length) {
        result.push(
            <tbody key="normal" className={b('rows')}>
                {rows}
            </tbody>,
        );
    }

    return result;
}

interface IStationTableProps {
    threads: IThreadRailroad[];
    id: number;
    goneThreadsAreOpened: boolean;
    event: StationEventList;

    whenDate?: DateRobot;
    whenSpecial?: StationDateSpecialValue;
    currentSubtype?: StationSubtype;
    className?: string;
}

function StationTable({
    threads,
    id,
    goneThreadsAreOpened,
    event,
    whenDate,
    whenSpecial,
    currentSubtype,
    className,
}: IStationTableProps): React.ReactElement {
    const dispatch = useDispatch();
    const tld = useSelector(state => state.tld);
    const language = useSelector(state => state.language);
    const now = useSelector(state => state.searchForm.time.now);
    const tableBodies = getTableBodies(
        threads,
        id,
        goneThreadsAreOpened,
        dispatch,
        event,
        tld,
        language,
        now,
        whenDate,
        whenSpecial,
        currentSubtype,
    );

    return <table className={b(undefined, className)}>{tableBodies}</table>;
}

export default React.memo(StationTable);
