import React, {
    ReactElement,
    memo,
    useMemo,
    useCallback,
    MouseEvent,
    RefObject,
} from 'react';
import B from 'bem-cn-lite';
import moment, {Moment} from 'moment';
import page from 'page';

import {HUMAN_SHORT} from '../../../lib/date/formats';

import IThreadRailroad from '../../../interfaces/state/station/IThreadRailroad';
import DateRobot from '../../../interfaces/date/DateRobot';
import StationEventList from '../../../interfaces/state/station/StationEventList';
import StopsType from '../../../interfaces/state/station/StopsType';
import {TransportType} from '../../../lib/transportType';
import StationDateSpecialValue from '../../../interfaces/date/StationDateSpecialValue';
import StationType from '../../../interfaces/state/station/StationType';
import StationSubtype from '../../../interfaces/state/station/StationSubtype';
import TransportSubtypeCode from '../../../interfaces/state/station/TransportSubtypeCode';
import IconGlyph from '../../../interfaces/components/IconGlyph';

import useSelector from '../../useSelector';
import useDispatch from '../../useDispatch';
import getShowGoneThreadsButtonText from './getShowGoneThreadsButtonText';
import {makeCacheable} from '../../../lib/__mocks__/cache';
import getThreadUrl from '../../../lib/url/getThreadUrl';
import getWhenDateText from '../getWhenDateText';

import {setGoneThreadsAreOpened} from '../../../actions/station';

import Icon from '../../Icon/Icon';
import TransportIcon from '../../TransportIcon/TransportIcon';
import Link from '../../Link';
import StationNoThreadsText from './StationNoThreadsText';
import ExceptString from '../../ExceptString/ExceptString';
import StationNextButton from '../../StationNextButton/StationNextButton';
import StationRunDays from '../../StationRunDays/StationRunDays';

import stationKeyset from '../../../i18n/station';

const b = B('StationTable');

type PatchedThread = IThreadRailroad & {
    // Добавляем инстанс момента для перфоманса
    eventMoment: Moment | null;
    showDate: boolean;
};

interface IStationTable {
    stationId: number;
    threads: IThreadRailroad[];
    whenDate: DateRobot | null;
    whenSpecial: StationDateSpecialValue | null;
    event: StationEventList;
    type: StationType;
    subtype: StationSubtype | null;
    goneThreadsAreOpened: boolean;

    className?: string;
    refToScrollAfterClickByNextButton?: RefObject<HTMLElement>;
}

const getOnThreadClick = makeCacheable(
    (threadUrl: string) => (e: MouseEvent<HTMLTableRowElement>) => {
        e.preventDefault();

        page.show(threadUrl);
    },
);

export default memo(StationTable);

function StationTable({
    stationId,
    threads,
    whenDate,
    whenSpecial,
    event,
    type,
    subtype,
    goneThreadsAreOpened,

    className,
    refToScrollAfterClickByNextButton,
}: IStationTable): ReactElement {
    const nowTime = useSelector(state => state.searchForm.time.now);
    const tld = useSelector(state => state.tld);
    const language = useSelector(state => state.language);
    const dispatch = useDispatch();

    const isAllDays = whenSpecial === StationDateSpecialValue.allDays;
    const nowMoment = moment(nowTime);
    const whenMoment = whenDate ? moment(whenDate) : null;

    const getThreadRow = useMemo(
        () =>
            makeCacheable((thread: PatchedThread) => {
                const {
                    eventMoment,
                    canonicalUid,
                    number,
                    eventDt: {time},
                    transportType,
                    isExpress,
                    isAeroExpress,
                    departure,
                    departureFrom,
                    platform,
                    stops,
                    deluxeTrainTitle,
                    transportSubtype,
                    daysText,
                    title,
                    showDate,
                    runDaysText,
                    exceptDaysText,
                } = thread;

                const threadUrl = getThreadUrl({
                    canonicalUid,
                    tld,
                    language,
                    ...(event === StationEventList.departure
                        ? {
                              stationFromId: stationId,
                              departureFromMoment: departureFrom
                                  ? moment.parseZone(departureFrom)
                                  : undefined,
                          }
                        : {
                              stationToId: stationId,
                              departure,
                          }),
                });

                const stopsText = ((): string | null => {
                    if (!stops) {
                        return null;
                    }

                    const {type: typeStop, text} = stops;

                    switch (typeStop) {
                        case StopsType.everywhere:
                            return stationKeyset('stops-everywhere');
                        case StopsType.nonstop:
                            return stationKeyset('stops-nonstop');
                        case StopsType.stops:
                            return text
                                ? `${stationKeyset('stops-stops')} ${text}`
                                : stationKeyset('stops-nonstop');
                    }
                })();

                const deluxeText =
                    transportType === TransportType.train
                        ? deluxeTrainTitle ?? transportSubtype?.title
                        : isAeroExpress
                        ? stationKeyset('aeroexpress')
                        : transportSubtype?.code !==
                          TransportSubtypeCode.suburban
                        ? transportSubtype?.title
                        : undefined;

                const formattedDaysText = (
                    <StationRunDays
                        daysText={daysText ?? null}
                        runDaysText={runDaysText ?? null}
                        exceptDaysText={exceptDaysText ?? null}
                        fontWeightForExcept="normal"
                    />
                );

                return (
                    <tr
                        className={b('threadRow')}
                        key={threadUrl}
                        onClick={getOnThreadClick(threadUrl)}
                    >
                        <td className={b('ceilDatetime')}>
                            {time}

                            {showDate && (
                                <div className={b('date')}>
                                    {eventMoment?.format(HUMAN_SHORT)}
                                </div>
                            )}
                        </td>

                        <td className={b('ceilIcon')}>
                            <TransportIcon
                                transportType={transportType}
                                isExpress={isExpress}
                                isAeroExpress={isAeroExpress}
                            />
                        </td>

                        <td className={b('ceilInfo')}>
                            <Link href={threadUrl}>{title}</Link>

                            <span className={b('number')}>{number}</span>

                            {Boolean(deluxeText) && (
                                <div className={b('deluxe')}>{deluxeText}</div>
                            )}

                            {!isAllDays && Boolean(platform) && (
                                <div className={b('platform')}>{platform}</div>
                            )}

                            {isAllDays && Boolean(daysText) && (
                                <div className={b('daysOfWalking')}>
                                    {formattedDaysText}
                                </div>
                            )}

                            {stopsText && (
                                <div className={b('stops')}>
                                    <ExceptString text={stopsText} />
                                </div>
                            )}
                        </td>
                    </tr>
                );
            }),
        [event, isAllDays, language, stationId, tld],
    );

    const [goneThreads, presentThreads] = threads.reduce<
        [PatchedThread[], PatchedThread[]]
    >(
        (arrayThreads, thread) => {
            const [gone, present] = arrayThreads;
            const datetime = thread.eventDt.datetime;
            const eventMoment = datetime ? moment.parseZone(datetime) : null;
            const showDate =
                eventMoment && whenMoment
                    ? eventMoment.format('DD') !== whenMoment.format('DD')
                    : false;
            const patchedThread: PatchedThread = {
                ...thread,
                eventMoment,
                showDate,
            };

            if (eventMoment && eventMoment.isBefore(nowMoment)) {
                gone.push({
                    ...patchedThread,
                    showDate: patchedThread.showDate || gone.length === 0,
                });
            } else {
                present.push({
                    ...patchedThread,
                    showDate:
                        patchedThread.showDate ||
                        (present.length === 0 && !goneThreadsAreOpened),
                });
            }

            return arrayThreads;
        },
        [[], []],
    );

    const onClickGoneButton = useCallback(() => {
        dispatch(setGoneThreadsAreOpened(!goneThreadsAreOpened));
    }, [dispatch, goneThreadsAreOpened]);

    const goneButton = useMemo(() => {
        const countGone = goneThreads.length;
        const countPresent = presentThreads.length;

        if (!countGone || !countPresent) {
            return null;
        }

        const text = getShowGoneThreadsButtonText(
            goneThreadsAreOpened,
            event,
            countGone,
            subtype,
        );

        return (
            <tr>
                <td
                    colSpan={3}
                    className={b('goneButton')}
                    onClick={onClickGoneButton}
                >
                    <span>{text}</span>

                    <Icon
                        className={b('goneIcon')}
                        glyph={IconGlyph.arrowDown}
                    />
                </td>
            </tr>
        );
    }, [
        event,
        goneThreads.length,
        goneThreadsAreOpened,
        onClickGoneButton,
        presentThreads.length,
        subtype,
    ]);

    const whenDateText = getWhenDateText(
        type,
        event,
        whenDate ?? undefined,
        whenSpecial ?? undefined,
        true,
    );

    return (
        <div
            className={b(
                {
                    goneThreadsAreOpened:
                        goneThreadsAreOpened || presentThreads.length === 0,
                    goneButtonExists: Boolean(goneButton),
                    subtype: subtype ?? '',
                },
                className,
            )}
        >
            {whenDateText && (
                <div className={b('whenDate')}>{whenDateText}</div>
            )}

            <div
                className={b('localTime', {
                    onAllDays: whenSpecial === StationDateSpecialValue.allDays,
                })}
            >
                <Icon className={b('localTimeIcon')} glyph={IconGlyph.alert2} />

                {stationKeyset('local-time-disclaimer')}
            </div>

            {goneThreads.length !== 0 || presentThreads.length !== 0 ? (
                <table className={b('table')}>
                    <tbody className={b('goneThreads')}>
                        {goneThreads.map(getThreadRow)}
                    </tbody>

                    <tbody className={b('goneButtonContainer')}>
                        {goneButton}
                    </tbody>

                    <tbody className={b('presentThreads')}>
                        {presentThreads.map(getThreadRow)}
                    </tbody>
                </table>
            ) : (
                <StationNoThreadsText className={b('noThreadsText')} isMobile />
            )}

            <StationNextButton
                className={b('nextButton')}
                refToScrollAfterClick={refToScrollAfterClickByNextButton}
            />
        </div>
    );
}
