import React, {PureComponent, createRef} from 'react';
import B from 'bem-cn-lite';
import {connect, batch} from 'react-redux';

import IStateStation from '../../interfaces/state/station/IStateStation';
import IStatePage from '../../interfaces/state/IStatePage';
import Tld from '../../interfaces/Tld';
import Lang from '../../interfaces/Lang';
import MapStateToProps from '../../interfaces/MapStateToProps';
import ITime from '../../interfaces/state/searchForm/ITime';
import IFlags from '../../interfaces/state/flags/IStateFlags';
import Dispatch from '../../interfaces/actions/Dispatch';
import StationSubtype from '../../interfaces/state/station/StationSubtype';
import StationTime from '../../interfaces/state/station/StationTime';
import StationType from '../../interfaces/state/station/StationType';
import IconGlyph from '../../interfaces/components/IconGlyph';
import IStopSuggestElement from '../../interfaces/lib/station/IStopSuggestElement';

import getMapsUrl from '../../lib/url/getMapsUrl';
import {reachGoal, reachGoalOnce} from '../../lib/yaMetrika';
import getTitleForPage from './getTitleForPage';
import getStationInformersUrl from '../../lib/url/getStationInformersUrl';
import getFilteredThreads from '../../lib/station/getFilteredThreads';
import isPlaneThreads from '../../lib/station/isPlaneThreads';
import isRailroadThreads from '../../lib/station/isRailroadThreads';
import getWhenDateText from './getWhenDateText';
import getFunctionForGetDataStationSuggest from '../../lib/station/getFunctionForGetDataStationSuggest';
import {makeCacheable} from '../../lib/cache';
import getRegexpForSuggestStops from '../../lib/station/getRegexpForSuggestStops';

import {
    changeFilterSearch,
    setTime,
    changeFilterStop,
    setStopText,
} from '../../actions/station';

import Link from '../Link';
import Icon from '../Icon/Icon';
import LoadingChunk from '../basic/LoadingChunk';
import StationTeaser from './StationTeaser/StationTeaser.desktop';
import DirectionTabs from './DirectionTabs';
import StationTable from './StationTable/StationTable.desktop';
import StationPlaneTable from './StationPlaneTable/StationPlaneTable.desktop';
import CityStations from './CityStations/CityStations.desktop';
import PageSubtypeTabs from './PageSubtypeTabs/PageSubtypeTabs.desktop';
import PopularDirections from './PopularDirections';
import DisclaimerText from './DisclaimerText';
import Direct from '../Direct/Direct';
import StationFilters from './StationFilters/StationFilters.desktop';
import StationNoThreadsText from './StationTable/StationNoThreadsText';
import Input from '../Input/Input';
import StationPlaneNextButton from './StationPlaneNextButton/StationPlaneNextButton';
import StationBusContentDesktop from '../StationBusContent/StationBusContent.desktop';
import StationWayToAirport from '../StationWayToAirport/StationWayToAirport';
import SuggestDesktop from '../SuggestDesktop/SuggestDesktop';

import stationKeyset from '../../i18n/station';

const b = B('StationPage');

const subtypeToDirectId = {
    [StationSubtype.suburban]: 'R-I-94177-46',
    [StationSubtype.train]: 'R-I-94177-47',
    [StationSubtype.plane]: 'R-I-94177-48',
    [StationSubtype.tablo]: 'R-I-94177-78',
};

// Нужно будет задать отдельный ИД-шник если захотим считать
const DEFAULT_DIRECT_ID = 'R-I-94177-47';

const LIMIT_STOP_SUGGEST = 5;

interface IStationPagePropsFromState {
    tld: Tld;
    language: Lang;
    flags: IFlags;
    pageFromState: IStatePage;
    time: ITime;
    station: IStateStation;
}

interface IStationPageProps extends IStationPagePropsFromState {
    dispatch: Dispatch;
}

const mapStateToProps: MapStateToProps<IStationPagePropsFromState> = ({
    tld,
    language,
    flags,
    page: pageFromState,
    searchForm,
    station,
}) => ({
    tld,
    language,
    flags,
    pageFromState,
    time: searchForm.time,
    station,
});

class StationPage extends PureComponent<IStationPageProps> {
    titleRef = createRef<HTMLHeadingElement>();

    getFunctionForGetDataStationSuggest = makeCacheable(
        getFunctionForGetDataStationSuggest,
    );

    getRegexpForSuggest = makeCacheable(getRegexpForSuggestStops);

    componentDidMount(): void {
        const {
            tld,
            language,
            station: {id, type, currentSubtype, event, teasers, subtypes},
        } = this.props;

        if (currentSubtype) {
            reachGoalOnce(`station_${currentSubtype}_${event}_page_shown`);
        }

        if (teasers.length) {
            reachGoalOnce('station_teasers_shown');
        }

        subtypes.forEach(subtype => {
            if (subtype !== currentSubtype) {
                reachGoalOnce(`station_alternative_type_${subtype}_shown`);
            }
        });

        if (
            getStationInformersUrl({
                id,
                tld,
                language,
                type,
                subtype: currentSubtype,
            })
        ) {
            reachGoalOnce('station_informer_link_shown');
        }
    }

    componentDidUpdate(prevProps: Readonly<IStationPageProps>): void {
        const {
            station: {currentSubtype, event, teasers, subtypes},
        } = this.props;

        if (
            currentSubtype &&
            currentSubtype !== prevProps.station.currentSubtype
        ) {
            reachGoalOnce(`station_${currentSubtype}_${event}_page_shown`);
        }

        if (teasers.length && teasers !== prevProps.station.teasers) {
            reachGoalOnce('station_teasers_shown');
        }

        if (subtypes !== prevProps.station.subtypes) {
            subtypes.forEach(subtype => {
                if (subtype !== currentSubtype) {
                    reachGoalOnce(`station_alternative_type_${subtype}_shown`);
                }
            });
        }
    }

    onMapClick = (): void => {
        reachGoal('station_map_click');
    };

    onClickPrint = (): void => {
        const {
            station: {currentSubtype},
        } = this.props;

        if (currentSubtype) {
            reachGoal(`station_${currentSubtype}_print_click`);
        }

        window.print();
    };

    onChangeStopText = (value: string): void => {
        const {dispatch} = this.props;

        dispatch(setStopText(value));
    };

    onResetStopFilter = (): void => {
        this.props.dispatch(changeFilterStop(null));
    };

    onClickStopElemet = (suggestElement: IStopSuggestElement): void => {
        const {dispatch, station} = this.props;

        const stop = station.stops.find(({id}) => id === suggestElement.id);

        if (stop) {
            dispatch(changeFilterStop(stop));
        }
    };

    onInformerLinkClick(): void {
        reachGoal('station_informer_link_click');
    }

    onSearchInputChange = (value: string): void => {
        const {
            dispatch,
            station: {type},
        } = this.props;

        if (value) {
            reachGoalOnce('station_search_input_used');
        } else {
            reachGoalOnce('station_search_input_emptied');
        }

        batch(() => {
            dispatch(changeFilterSearch(value));

            if (value && type === StationType.plane) {
                // При поиске на странице аэропорта переключаемся на поиск по всем интервалам времени
                dispatch(setTime(StationTime.all));
            }
        });
    };

    getTeasers(): React.ReactElement[] | null {
        const {
            station: {teasers},
        } = this.props;

        if (!teasers.length) {
            return null;
        }

        return teasers.map(teaser => {
            const {title, content, mobileContent, type, id, isOpened} = teaser;

            return (
                <StationTeaser
                    className={b('teaser')}
                    key={type}
                    id={id}
                    isOpened={isOpened}
                    title={title}
                    content={content}
                    mobileContent={mobileContent ?? null}
                />
            );
        });
    }

    getDateAndSearchBlock(): React.ReactElement {
        const {
            station: {type, currentSubtype, stopText, stops, search},
        } = this.props;

        const isPlanePage = type === StationType.plane;
        const isBusWaterPage =
            type === StationType.bus || type === StationType.water;

        let searchDefaultText;

        switch (currentSubtype) {
            case StationSubtype.suburban:
                searchDefaultText = stationKeyset('search-suburban');
                break;
            case StationSubtype.train:
            case StationSubtype.tablo:
                searchDefaultText = stationKeyset('search-train');
                break;
            case StationSubtype.plane:
                searchDefaultText = stationKeyset('search-plane');
                break;
            case StationSubtype.schedule:
                searchDefaultText = stationKeyset('search-station-placeholder');
                break;

            default:
                searchDefaultText = stationKeyset('search-default');
        }

        return (
            <div
                className={b('dateAndSearchBlock', {
                    notPlanePage: !isPlanePage,
                })}
            >
                <StationFilters className={b('stationFilters')} />

                {isBusWaterPage ? (
                    <SuggestDesktop
                        className={b('searchInput')}
                        value={stopText}
                        onChange={this.onChangeStopText}
                        getData={this.getFunctionForGetDataStationSuggest(
                            stops,
                            LIMIT_STOP_SUGGEST,
                        )}
                        onClickElement={this.onClickStopElemet}
                        onReset={this.onResetStopFilter}
                        placeholder={stationKeyset(
                            'search-station-placeholder',
                        )}
                        regExpForHighlightingElements={this.getRegexpForSuggest(
                            stopText,
                        )}
                        limit={LIMIT_STOP_SUGGEST}
                    />
                ) : (
                    <Input
                        className={b('searchInput')}
                        onChange={this.onSearchInputChange}
                        value={search}
                        placeholder={searchDefaultText}
                        autoComplete={false}
                        withSearchIcon
                        withReset
                    />
                )}
            </div>
        );
    }

    renderTable(): React.ReactElement | null {
        const {
            station: {
                type,
                search,
                threads,
                currentSubtype,
                companiesById,
                terminals,
                trusted,
                whenSpecial,
                whenDate,
                event,
                id,
                terminalName,
                time,
                goneThreadsAreOpened,
            },
        } = this.props;

        if (type === StationType.bus || type === StationType.water) {
            return <StationBusContentDesktop />;
        }

        if (threads.length === 0) {
            return (
                <StationNoThreadsText
                    className={b('noThreads')}
                    isMobile={false}
                />
            );
        }

        const threadsForDisplay = getFilteredThreads({
            threads,
            search,
            isMobile: false,
            terminalName,
            time: time ?? null,
            event,
            companiesById,
            whenDate,
        });

        if (threadsForDisplay.length === 0) {
            return (
                <div className={b('noThreadsForFilter')}>
                    {stationKeyset('no-threads-for-filter')}
                </div>
            );
        }

        if (type === StationType.plane && isPlaneThreads(threadsForDisplay)) {
            return (
                <StationPlaneTable
                    threads={threadsForDisplay}
                    companiesById={companiesById}
                    terminals={terminals}
                    trusted={trusted}
                    whenSpecial={whenSpecial}
                    event={event}
                />
            );
        }

        if (isRailroadThreads(threadsForDisplay)) {
            return (
                <StationTable
                    id={id}
                    threads={threadsForDisplay}
                    goneThreadsAreOpened={goneThreadsAreOpened}
                    whenDate={whenDate}
                    whenSpecial={whenSpecial}
                    event={event}
                    currentSubtype={currentSubtype}
                />
            );
        }

        return null;
    }

    getPrintButton(): React.ReactElement {
        return (
            <div className={b('printButton')} onClick={this.onClickPrint}>
                <Icon className={b('printIcon')} glyph={IconGlyph.printIcon} />

                <span className={b('printText')}>{stationKeyset('print')}</span>
            </div>
        );
    }

    getTopTabs(): React.ReactElement {
        const {currentSubtype} = this.props.station;

        return (
            <div className={b('topRow')}>
                {currentSubtype && <PageSubtypeTabs className={b('tabs')} />}
                {this.getPrintButton()}
            </div>
        );
    }

    renderLoading(): React.ReactElement {
        return <LoadingChunk />;
    }

    pageIsFetching(): boolean {
        return this.props.pageFromState.fetching !== null;
    }

    render(): React.ReactElement {
        const {station, tld, language} = this.props;

        const {
            title,
            fullTitle,
            fullTitleDative,
            hasPopularTitle,
            address,
            latitude,
            longitude,
            subway,
            currentSubtype,
            subtypes,
            whenSpecial,
            whenDate,
            event,
            id,
            type,
            cityData,
            settlement,
            popularDirections,
            terminalName,
            terminals,
            phones,
        } = station;

        const isPlanePage = type === StationType.plane;
        const isBusWaterPage =
            type === StationType.bus || type === StationType.water;
        const isRailroadPage = type === StationType.railroad;
        const directSidebarId =
            (currentSubtype && subtypeToDirectId[currentSubtype]) ||
            DEFAULT_DIRECT_ID;
        const terminal = terminals.find(
            terminalObj => terminalObj.name === terminalName,
        );
        const pageIsFetching = this.pageIsFetching();

        if (pageIsFetching && !id) {
            return (
                <div className={b({isLoading: true})}>
                    <div className={b('centralArea')}>
                        <div className={b('mainContent')}>
                            {this.renderLoading()}
                        </div>
                    </div>
                    <div className={b('sidebar')}>
                        <Direct
                            className={b('directSidebar')}
                            blockId={directSidebarId}
                            updateIfLoadedAnotherPage={false}
                        />
                    </div>
                </div>
            );
        }

        const showTopTabs = !isBusWaterPage && subtypes.length > 1;
        const informerLink = getStationInformersUrl({
            id,
            tld,
            language,
            type,
            subtype: currentSubtype,
        });

        const whenDateText = getWhenDateText(
            type,
            event,
            whenDate,
            whenSpecial,
        );

        const joinedPhones =
            isBusWaterPage &&
            phones &&
            phones
                .reduce<string[]>((result, phone) => {
                    return result.concat(
                        phone
                            .split(',')
                            .map(val => val.trim())
                            .filter(Boolean),
                    );
                }, [])
                .join(', ');

        return (
            <div className={b({type, isLoading: pageIsFetching})}>
                <div className={b('centralArea')}>
                    <div className={b('mainContent')}>
                        {showTopTabs && this.getTopTabs()}

                        <div className={b('titleContainer')}>
                            <h1 className={b('title')} ref={this.titleRef}>
                                {getTitleForPage({
                                    type,
                                    title,
                                    hasPopularTitle,
                                    event,
                                    currentSubtype,
                                    fullTitle,
                                    fullTitleDative,
                                    terminal,
                                    settlementTitle: settlement?.title,
                                })}
                            </h1>
                            {!showTopTabs &&
                                isRailroadPage &&
                                this.getPrintButton()}
                        </div>

                        <div className={b('addressAndPrintBlock')}>
                            <div className={b('addressBlock')}>
                                {typeof longitude !== 'undefined' &&
                                    typeof latitude !== 'undefined' && (
                                        <Link
                                            className={b('addressLink', {
                                                withAddress: Boolean(address),
                                            })}
                                            href={getMapsUrl(
                                                longitude,
                                                latitude,
                                                tld,
                                                language,
                                            )}
                                            onClick={this.onMapClick}
                                        >
                                            <Icon
                                                className={b('addressIcon')}
                                                glyph={IconGlyph.mapPoint}
                                            />
                                            {address ||
                                                stationKeyset(
                                                    'map-default-text',
                                                )}
                                        </Link>
                                    )}

                                {subway && (
                                    <span className={b('subway')}>
                                        {subway}
                                    </span>
                                )}

                                {isPlanePage && (
                                    <StationWayToAirport
                                        className={b('aeroexpressLink')}
                                        showIcon
                                    />
                                )}
                            </div>

                            {(isPlanePage || isBusWaterPage) &&
                                this.getPrintButton()}
                        </div>

                        {joinedPhones && (
                            <div className={b('phones')}>{joinedPhones}</div>
                        )}

                        <div className={b('teasers')}>{this.getTeasers()}</div>

                        {this.getDateAndSearchBlock()}

                        {currentSubtype !== StationSubtype.schedule && (
                            <DirectionTabs className={b('directionTabs')} />
                        )}

                        <div className={b('dateAndTimeDisclaimer')}>
                            {whenDateText && (
                                <div className={b('whenDate')}>
                                    {whenDateText}
                                </div>
                            )}

                            <div className={b('localTimeDisclaimer')}>
                                <Icon
                                    className={b('localTimeIcon')}
                                    glyph={IconGlyph.alert2}
                                />
                                {stationKeyset('local-time-disclaimer')}
                            </div>
                        </div>

                        <div className={b('tableContainer')}>
                            {pageIsFetching && (
                                <LoadingChunk className={b('tablePreloader')} />
                            )}

                            <div className={b('tableWrapper')}>
                                {this.renderTable()}
                            </div>
                        </div>

                        <StationPlaneNextButton
                            className={b('nextButton')}
                            refToScroll={this.titleRef}
                        />
                    </div>

                    <DisclaimerText className={b('disclaimer')} />

                    {informerLink && (
                        <div className={b('informerLink')}>
                            <Link
                                href={informerLink}
                                onClick={this.onInformerLinkClick}
                            >
                                <Icon
                                    className={b('serviceIcon')}
                                    glyph={IconGlyph.serviceDescription}
                                />
                                {stationKeyset('informer-link')}
                            </Link>
                        </div>
                    )}
                </div>

                <div className={b('sidebar')}>
                    <Direct
                        className={b('directSidebar')}
                        blockId={directSidebarId}
                        updateIfLoadedAnotherPage={false}
                    />

                    {cityData && (
                        <CityStations
                            type={type}
                            currentSubtype={currentSubtype}
                            cityData={cityData}
                            className={b('cityStations')}
                        />
                    )}

                    <PopularDirections
                        event={event}
                        popularDirections={popularDirections}
                        title={
                            type === StationType.water
                                ? stationKeyset('popular-directions-water')
                                : undefined
                        }
                        className={b('popularDirections')}
                    />
                </div>
            </div>
        );
    }
}

export default connect(mapStateToProps)(StationPage);
