import StationEventList from '../../interfaces/state/station/StationEventList';
import StationSubtype from '../../interfaces/state/station/StationSubtype';
import ITerminal from '../../interfaces/state/station/ITerminal';
import StationType from '../../interfaces/state/station/StationType';

import titleHasSettlementTitle from '../../lib/station/titleHasSettlementTitle';

import stationKeyset from '../../i18n/station';
import stationPlaneKeyset from '../../i18n/station-plane-seo';

const lastDotRegExp = /\.$/;

interface IGetTitleForPage {
    title: string;
    hasPopularTitle: boolean;
    event: StationEventList;
    type: StationType;

    currentSubtype?: StationSubtype;
    fullTitle?: string;
    fullTitleDative?: string;
    terminal?: ITerminal;
    settlementTitle?: string;
}

export default function getTitleForPage({
    title,
    hasPopularTitle,
    event,
    type,
    currentSubtype,
    fullTitle,
    fullTitleDative,
    terminal,
    settlementTitle,
}: IGetTitleForPage): string {
    const eventForTitle =
        event === StationEventList.arrival
            ? stationKeyset('arrival')
            : stationKeyset('departure');
    const stationName = fullTitle || title;
    const stationNameDative = fullTitleDative || stationName;

    switch (currentSubtype) {
        case StationSubtype.suburban:
            if (hasPopularTitle) {
                if (settlementTitle) {
                    return stationKeyset('title-suburban-popular', {
                        title: stationName,
                        cityName: settlementTitle,
                    });
                }

                return stationKeyset('title-suburban-popular-no-city', {
                    title: stationName,
                });
            }

            return stationKeyset('title-suburban', {
                titleDative: stationNameDative,
            });
        case StationSubtype.train:
            if (hasPopularTitle) {
                if (settlementTitle) {
                    return `${stationKeyset('title-train-popular', {
                        title: stationName,
                        cityName: settlementTitle,
                    })}. ${eventForTitle}`;
                }

                return `${stationKeyset('title-train-popular-no-city', {
                    title: stationName.replace(lastDotRegExp, ''),
                })}. ${eventForTitle}`;
            }

            return `${stationKeyset('title-train', {
                titleDative: stationNameDative.replace(lastDotRegExp, ''),
            })}. ${eventForTitle}`;
        case StationSubtype.plane:
            if (terminal) {
                if (
                    event === StationEventList.departure &&
                    title === settlementTitle
                ) {
                    return stationPlaneKeyset(
                        'h1-with-terminal-departure-same-names',
                        {
                            airportTitle: title,
                            cityTitle: settlementTitle || '',
                            terminal: terminal.name,
                        },
                    );
                }

                if (
                    event === StationEventList.arrival &&
                    title === settlementTitle
                ) {
                    return stationPlaneKeyset(
                        'h1-with-terminal-arrival-same-names',
                        {
                            airportTitle: title,
                            cityTitle: settlementTitle || '',
                            terminal: terminal.name,
                        },
                    );
                }

                if (
                    event === StationEventList.departure &&
                    title !== settlementTitle
                ) {
                    return stationPlaneKeyset(
                        'h1-with-terminal-departure-different-names',
                        {
                            airportTitle: title,
                            cityTitle: settlementTitle || '',
                            terminal: terminal.name,
                        },
                    );
                }

                if (
                    event === StationEventList.arrival &&
                    title !== settlementTitle
                ) {
                    return stationPlaneKeyset(
                        'h1-with-terminal-arrival-different-names',
                        {
                            airportTitle: title,
                            cityTitle: settlementTitle || '',
                            terminal: terminal.name,
                        },
                    );
                }
            }

            if (event === StationEventList.arrival) {
                if (title === settlementTitle) {
                    return stationPlaneKeyset('h1-arrival-same-names', {
                        airportTitle: title,
                        cityTitle: settlementTitle || '',
                    });
                }

                return stationPlaneKeyset('h1-arrival-different-names', {
                    airportTitle: title,
                    cityTitle: settlementTitle || '',
                });
            }

            if (title === settlementTitle) {
                return stationPlaneKeyset('h1-departure-same-names', {
                    airportTitle: title,
                    cityTitle: settlementTitle || '',
                });
            }

            return stationPlaneKeyset('h1-departure-different-names', {
                airportTitle: title,
                cityTitle: settlementTitle || '',
            });
        case StationSubtype.tablo:
            if (hasPopularTitle) {
                if (settlementTitle) {
                    return `${stationKeyset('title-tablo-popular', {
                        title: stationName,
                        cityName: settlementTitle,
                    })}. ${eventForTitle}`;
                }

                return `${stationKeyset('title-tablo-popular-no-city', {
                    title: stationName.replace(lastDotRegExp, ''),
                })}. ${eventForTitle}`;
            }

            return `${stationKeyset('title-tablo', {
                titleDative: stationNameDative.replace(lastDotRegExp, ''),
            })}. ${eventForTitle}`;
    }

    if (type === StationType.bus || type === StationType.water) {
        const titleForPage =
            settlementTitle &&
            !titleHasSettlementTitle(stationName, settlementTitle)
                ? `${stationName} (${settlementTitle})`
                : stationName;

        switch (type) {
            case StationType.bus:
                return stationKeyset('title-bus', {title: titleForPage});
            case StationType.water:
                return stationKeyset('title-water', {title: titleForPage});
        }
    }

    return `${stationKeyset('title-unknown', {
        titleDative: stationNameDative.replace(lastDotRegExp, ''),
    })}. ${eventForTitle}`;
}
