import React, {
    memo,
    ReactElement,
    useCallback,
    useEffect,
    useMemo,
    useRef,
    PropsWithChildren,
} from 'react';
import B from 'bem-cn-lite';
import upperFirst from 'lodash/upperFirst';

import StationType from '../../interfaces/state/station/StationType';
import StationSubtype from '../../interfaces/state/station/StationSubtype';
import StationEventList from '../../interfaces/state/station/StationEventList';
import IconGlyph from '../../interfaces/components/IconGlyph';
import {TransportType} from '../../lib/transportType';

import {reachGoal, reachGoalOnce} from '../../lib/yaMetrika';
import useSelector from '../useSelector';
import getMapsUrl from '../../lib/url/getMapsUrl';
import getTitleForPage from './getTitleForPage';
import getFilteredThreads from '../../lib/station/getFilteredThreads';
import isPlaneThreads from '../../lib/station/isPlaneThreads';
import isRailroadThreads from '../../lib/station/isRailroadThreads';
import scrollWindow from '../../../client/lib/scrollWindow';
import {stationUrl} from '../../lib/url/stationUrl';
import {useFlags} from '../../hooks/useFlags';
import useDispatch from '../useDispatch';

import {changeFilterStop} from '../../actions/station';

import Icon from '../Icon/Icon';
import Link from '../Link';
import PageSubtypeTabs from './PageSubtypeTabs/PageSubtypeTabs.mobile';
import LoadingChunk from '../basic/LoadingChunk';
import StationFilters from './StationFilters/StationFilters.mobile';
import StationTable from './StationTable/StationTable.mobile';
import StationPlaneTable from './StationPlaneTable/StationPlaneTable.mobile';
import StationTeaser from './StationTeaser/StationTeaser.mobile';
import CityStations from './CityStations/CityStations.mobile';
import PopularDirections from './PopularDirections';
import DisclaimerText from './DisclaimerText';
import Direct from '../Direct/Direct';
import StationPlaneNextButton from './StationPlaneNextButton/StationPlaneNextButton';
import PlaneDirectionTabs from './PlaneDirectionTabs';
import StationWayToAirport from '../StationWayToAirport/StationWayToAirport';
import StationBusContent from '../StationBusContent/StationBusContent.mobile';
import NativeHorizontScroll from '../NativeHorizontScroll/NativeHorizontScroll';
import TransportIconInCircle from '../TransportIconInCircle/TransportIconInCircle';

import stationKeyset from '../../i18n/station';

const b = B('StationPage');

export default memo(StationPage);

const DIRECT_ID_STATION = 'R-I-239082-69';
const DIRECT_ID_STATION_EXPERIMENT = 'R-I-239082-70';

const transportTypeByStationType = {
    [StationType.railroad]: TransportType.train,
    [StationType.plane]: TransportType.plane,
    [StationType.bus]: TransportType.bus,
    [StationType.water]: TransportType.water,
};

function StationPage(): ReactElement | null {
    const tld = useSelector(state => state.tld);
    const language = useSelector(state => state.language);
    const page = useSelector(state => state.page);
    const station = useSelector(state => state.station);
    const flags = useFlags();

    const dispatch = useDispatch();

    const pageRef = useRef<HTMLDivElement>(null);
    const refTitle = useRef<HTMLHeadingElement>(null);

    const pageIsFetching = Boolean(page.fetching);
    const {
        id,
        title,
        fullTitle,
        fullTitleDative,
        hasPopularTitle,
        address,
        longitude,
        latitude,
        subway,
        currentSubtype,
        mainSubtype,
        type,
        event,
        whenDate,
        whenSpecial,
        goneThreadsAreOpened,
        settlement,
        threads,
        cityData,
        popularDirections,
        companiesById,
        terminals,
        teasers,
        search,
        stop,
        terminalName,
        time,
    } = station;

    const isBusWaterType =
        type === StationType.bus || type === StationType.water;

    useEffect(() => {
        if (currentSubtype) {
            reachGoalOnce(`station_${currentSubtype}_${event}_page_shown`);
        }
    }, [currentSubtype, event]);

    useEffect(() => {
        if (teasers.length) {
            reachGoalOnce('station_teasers_shown');
        }
    }, [teasers]);

    const onMapClick = useCallback(() => {
        reachGoal('station_map_click');
    }, []);

    const onStationLinkClick = useCallback(() => {
        if (pageRef.current) {
            scrollWindow(pageRef);
        }
    }, []);

    const onClickResetFilterByStation = useCallback(() => {
        dispatch(changeFilterStop(null));
    }, [dispatch]);

    const terminal = terminals.find(
        terminalObj => terminalObj.name === terminalName,
    );

    const wayToAirportLink = <StationWayToAirport showIcon shortText />;

    const teaserBlocks = useMemo(() => {
        if (!teasers.length) {
            return null;
        }

        return (
            <div className={b('teasers')}>
                {teasers.map((teaser, index) => (
                    <StationTeaser
                        key={`${teaser.id}-${index}`}
                        className={b('teaser')}
                        isOpened={teaser.isOpened}
                        title={teaser.title}
                        content={teaser.content}
                        mobileContent={teaser.mobileContent ?? null}
                        id={teaser.id}
                    />
                ))}
            </div>
        );
    }, [teasers]);

    const tabsBlock = useMemo(() => {
        if (!currentSubtype || isBusWaterType) {
            return null;
        }

        return (
            <NativeHorizontScroll className={b('tabsWrapper')}>
                <div className={b('tabsContainer')}>
                    {currentSubtype === StationSubtype.plane ? (
                        <div className={b('planeTabs')}>
                            <PlaneDirectionTabs
                                className={b('tabs')}
                                stationId={id}
                                currentSubtype={currentSubtype}
                                mainSubtype={mainSubtype}
                                type={type}
                                date={whenDate}
                                special={whenSpecial}
                                event={event}
                                terminalName={terminalName}
                            />

                            {wayToAirportLink}
                        </div>
                    ) : (
                        <PageSubtypeTabs className={b('tabs')} />
                    )}
                </div>
            </NativeHorizontScroll>
        );
    }, [
        currentSubtype,
        event,
        id,
        isBusWaterType,
        mainSubtype,
        terminalName,
        type,
        wayToAirportLink,
        whenDate,
        whenSpecial,
    ]);

    const {__everlastingStationTouchExperiment} = flags;

    const directId = __everlastingStationTouchExperiment
        ? DIRECT_ID_STATION_EXPERIMENT
        : DIRECT_ID_STATION;

    const table = useMemo(() => {
        if (type === StationType.bus || type === StationType.water) {
            return (
                <TableContainer
                    pageIsFetching={pageIsFetching}
                    withoutBackgroundIfNotLoading
                >
                    <StationBusContent
                        refToScrollAfterClickNextButton={refTitle}
                    />
                </TableContainer>
            );
        }

        const threadsForDisplay = getFilteredThreads({
            threads,
            search,
            isMobile: true,
            terminalName,
            time: time ?? null,
            event,
            companiesById,
            whenDate,
        });

        if (
            (search || terminalName || time) &&
            threadsForDisplay.length === 0
        ) {
            return (
                <TableContainer pageIsFetching={pageIsFetching} empty>
                    {stationKeyset('no-threads-for-filter')}
                </TableContainer>
            );
        }

        if (type === StationType.plane && isPlaneThreads(threadsForDisplay)) {
            return (
                <TableContainer pageIsFetching={pageIsFetching}>
                    <StationPlaneTable
                        threads={threadsForDisplay}
                        companiesById={companiesById}
                        event={event}
                        terminals={terminals}
                        whenSpecial={whenSpecial}
                        whenDate={whenDate}
                        type={type}
                    />

                    <StationPlaneNextButton
                        className={b('nextButton')}
                        refToScroll={refTitle}
                    />
                </TableContainer>
            );
        }

        if (
            type === StationType.railroad &&
            isRailroadThreads(threadsForDisplay)
        ) {
            return (
                <TableContainer pageIsFetching={pageIsFetching}>
                    <StationTable
                        className={b('table')}
                        stationId={id}
                        threads={threadsForDisplay}
                        whenDate={whenDate ?? null}
                        whenSpecial={whenSpecial ?? null}
                        event={event}
                        type={type}
                        subtype={currentSubtype ?? null}
                        goneThreadsAreOpened={goneThreadsAreOpened}
                        refToScrollAfterClickByNextButton={refTitle}
                    />
                </TableContainer>
            );
        }
    }, [
        companiesById,
        currentSubtype,
        event,
        goneThreadsAreOpened,
        id,
        pageIsFetching,
        search,
        terminalName,
        terminals,
        threads,
        time,
        type,
        whenDate,
        whenSpecial,
    ]);

    if (pageIsFetching && !id) {
        return <LoadingChunk />;
    }

    return (
        <div className={b({isLoading: pageIsFetching, type})} ref={pageRef}>
            <div className={b('header')}>
                <div className={b('titleWithIcon')}>
                    <h1 className={b('title')} ref={refTitle}>
                        {getTitleForPage({
                            type,
                            title,
                            hasPopularTitle,
                            event,
                            currentSubtype,
                            fullTitle,
                            fullTitleDative,
                            terminal,
                            settlementTitle: settlement?.title,
                        })}
                    </h1>

                    <TransportIconInCircle
                        transportType={transportTypeByStationType[type]}
                        className={b('titleIcon')}
                    />
                </div>

                <div className={b('addressBlock')}>
                    {typeof longitude !== 'undefined' &&
                        typeof latitude !== 'undefined' && (
                            <Link
                                className={b('addressLink')}
                                href={getMapsUrl(
                                    longitude,
                                    latitude,
                                    tld,
                                    language,
                                )}
                                onClick={onMapClick}
                            >
                                <Icon
                                    className={b('addressIcon')}
                                    glyph={IconGlyph.mapPoint}
                                />

                                {address || stationKeyset('map-default-text')}
                            </Link>
                        )}

                    {subway && <span className={b('subway')}>{subway}</span>}
                </div>

                {tabsBlock}

                {Boolean(tabsBlock) && Boolean(teaserBlocks) && (
                    <div className={b('divider')} />
                )}

                {teaserBlocks}
            </div>

            <StationFilters className={b('filters')} />

            {stop && (
                <div className={b('stationFilter')}>
                    <span className={b('stationFilterName')}>
                        {stationKeyset('to-station-short', {title: stop.title})}
                    </span>

                    <span
                        className={b('stationFilterResetContainer')}
                        onClick={onClickResetFilterByStation}
                    >
                        <Icon
                            className={b('stationFilterReset')}
                            glyph={IconGlyph.crossBold}
                        />
                    </span>
                </div>
            )}

            {table}

            <DisclaimerText className={b('disclaimer')} />

            {currentSubtype === StationSubtype.train && (
                <Link
                    href={stationUrl({
                        id,
                        tld,
                        language,
                        type,
                        subtype: currentSubtype,
                        mainSubtype,
                        isMobile: true,
                        date: whenDate,
                        special: whenSpecial,
                        event:
                            event === StationEventList.arrival
                                ? StationEventList.departure
                                : StationEventList.arrival,
                        terminalName,
                        time,
                        search,
                    })}
                >
                    {stationKeyset(
                        event === StationEventList.arrival
                            ? 'to-departure-train-link'
                            : 'to-arrival-train-link',
                        {
                            title: upperFirst(fullTitle || title),
                        },
                    )}
                </Link>
            )}

            <CityStations
                type={type}
                currentSubtype={currentSubtype}
                cityData={cityData}
                className={b('cityStations')}
                onLinkClick={onStationLinkClick}
            />

            <PopularDirections
                event={event}
                popularDirections={popularDirections}
                title={
                    type === StationType.water
                        ? stationKeyset('popular-directions-water')
                        : undefined
                }
                className={b('popularDirections')}
            />

            <Direct blockId={directId} className={b('bottomDirect')} />
        </div>
    );
}

interface ITableContainer {
    pageIsFetching: boolean;

    empty?: boolean;
    withoutBackgroundIfNotLoading?: boolean;
}

function TableContainer({
    empty,
    pageIsFetching,
    children,
    withoutBackgroundIfNotLoading,
}: PropsWithChildren<ITableContainer>): ReactElement {
    return (
        <div
            className={b('tableContainer', {
                empty,
                withoutBackgroundIfNotLoading,
            })}
        >
            {pageIsFetching && <LoadingChunk className={b('tablePreloader')} />}

            <div className={b('tableWrapper')}>{children}</div>
        </div>
    );
}
