import React, {ReactElement, memo, Fragment} from 'react';
import B from 'bem-cn-lite';

import {CHAR_NBSP} from '../../lib/stringUtils';

import IStationScheduleBlock from '../../interfaces/state/station/IStationScheduleBlock';
import Lang from '../../interfaces/Lang';
import Tld from '../../interfaces/Tld';
import DateRobot from '../../interfaces/date/DateRobot';
import IStationStopFilter from '../../interfaces/state/station/IStationStopFilter';

import getScheduleThreads from '../../lib/station/getScheduleThreads';
import getThreadUrl from '../../lib/url/getThreadUrl';
import getScheduleThreadsAsTable from '../../lib/station/getScheduleThreadsAsTable';
import insertMarkupIntoKey from '../../i18n/_utils/insertMarkupIntoKey';

import TransportIcon from '../TransportIcon/TransportIcon';
import StationRunDays from '../StationRunDays/StationRunDays';
import Link from '../Link';
import StationThreadLink from '../StationThreadLink/StationThreadLink';

import stationKeyset from '../../i18n/station';

const b = B('StationScheduleBlock');

interface IStationScheduleBlockEl {
    scheduleBlock: IStationScheduleBlock;
    isAllDays: boolean;
    tld: Tld;
    language: Lang;
    stationFromId: number;
    whenDate: DateRobot | null;
    stopFilter: IStationStopFilter | null;

    className?: string;
}

export default memo(StationScheduleBlock);

// Применена ужасающая своей дебильностью верстка таблицей, потому что так нужно для SEO (не спрашивайте почему)
function StationScheduleBlock({
    scheduleBlock,
    isAllDays,
    tld,
    language,
    stationFromId,
    whenDate,
    stopFilter,

    className,
}: IStationScheduleBlockEl): ReactElement | null {
    const {
        title,
        number,
        transportType,
        comment,
        schedule: schedules,
    } = scheduleBlock;

    const threadBlocks = schedules
        .map((schedule, index) => {
            // index должен оставаться постоянным для блока с нитками, иначе будут проблемы с перфомансом при фильтрации по станции
            const {
                threads: allThreads,
                daysText,
                runDaysText,
                exceptDaysText,
            } = schedule;

            const {threads, intervalThreads} = getScheduleThreads(
                allThreads,
                stopFilter,
            );

            if (threads.length === 0 && intervalThreads.length === 0) {
                return null;
            }

            const threadsTable = getScheduleThreadsAsTable(threads, 18);
            const threadsTableOneRow = !threadsTable.some(
                ({threads: threadsColumn}) => threadsColumn.length > 1,
            );

            return (
                <Fragment key={index}>
                    {isAllDays && (
                        <tr>
                            <td className={b('runDays')}>
                                <StationRunDays
                                    daysText={daysText ?? null}
                                    runDaysText={runDaysText ?? null}
                                    exceptDaysText={exceptDaysText ?? null}
                                />
                            </td>
                        </tr>
                    )}

                    {/* Интервальные нитки */}
                    {intervalThreads.map(intervalThread => {
                        const {
                            interval: {beginTime, endTime, density},
                            canonicalUid,
                        } = intervalThread;

                        const url = getThreadUrl({
                            canonicalUid,
                            tld,
                            language,
                            stationFromId,
                            departureFromDateRobot: whenDate ?? undefined,
                        });

                        return (
                            <tr
                                className={b('intervalScheduleTr')}
                                key={canonicalUid}
                            >
                                <td className={b('intervalScheduleTd')}>
                                    <Link
                                        href={url}
                                        className={b('intervalScheduleLink')}
                                    >
                                        {stationKeyset('interval-thread-time', {
                                            beginTime,
                                            endTime,
                                        })}
                                    </Link>
                                    {` ${density}`}

                                    {comment && (
                                        <span className={b('comment')}>
                                            {' '}
                                            {comment
                                                .trim()
                                                .replace(' ', CHAR_NBSP)}
                                        </span>
                                    )}
                                </td>
                            </tr>
                        );
                    })}

                    {/* Таблица с нитками */}
                    {threadsTable.length > 0 && (
                        <tr className={b('tableScheduleTr')}>
                            <td>
                                <div
                                    className={b('tableScheduleColumns', {
                                        oneRow: threadsTableOneRow,
                                    })}
                                >
                                    {threadsTable.map(
                                        ({
                                            threads: threadsColumn,
                                            interval,
                                        }) => (
                                            <div
                                                className={b(
                                                    'tableScheduleColumn',
                                                )}
                                                key={interval}
                                            >
                                                {threadsColumn.map(
                                                    (thread, indexThread) => (
                                                        <StationThreadLink
                                                            className={b(
                                                                'threadLink',
                                                            )}
                                                            key={
                                                                thread.canonicalUid
                                                            }
                                                            thread={thread}
                                                            tld={tld}
                                                            language={language}
                                                            stationFromId={
                                                                stationFromId
                                                            }
                                                            prevThread={
                                                                threadsColumn[
                                                                    indexThread -
                                                                        1
                                                                ]
                                                            }
                                                        />
                                                    ),
                                                )}
                                            </div>
                                        ),
                                    )}
                                </div>
                            </td>
                        </tr>
                    )}
                </Fragment>
            );
        })
        .filter(Boolean);

    if (threadBlocks.length === 0) {
        return null;
    }

    return (
        <tbody className={b(undefined, className)}>
            <tr className={b('titleTr')}>
                <td className={b('titleTd')}>
                    <div className={b('titleContainer')}>
                        <TransportIcon
                            className={b('transportIcon', {transportType})}
                            transportType={transportType}
                        />

                        {number.length > 0 && (
                            <span className={b('number')}>№ {number}</span>
                        )}

                        <span className={b('name')}>{title}</span>
                    </div>

                    {stopFilter && (
                        <div className={b('stopFilter')}>
                            {insertMarkupIntoKey(stationKeyset, 'to-station', {
                                title: (
                                    <span
                                        className={b('stopFilterTitle')}
                                        key="stopFilterTitle"
                                    >
                                        {stopFilter.title}
                                    </span>
                                ),
                            })}
                        </div>
                    )}
                </td>
            </tr>

            {threadBlocks}
        </tbody>
    );
}
