import React, {ReactElement, memo, Fragment} from 'react';
import B from 'bem-cn-lite';

import {CHAR_NBSP} from '../../lib/stringUtils';

import IStationScheduleBlock from '../../interfaces/state/station/IStationScheduleBlock';
import Lang from '../../interfaces/Lang';
import Tld from '../../interfaces/Tld';
import DateRobot from '../../interfaces/date/DateRobot';
import IStationStopFilter from '../../interfaces/state/station/IStationStopFilter';

import getScheduleThreads from '../../lib/station/getScheduleThreads';
import getThreadUrl from '../../lib/url/getThreadUrl';

import TransportIcon from '../TransportIcon/TransportIcon';
import StationRunDays from '../StationRunDays/StationRunDays';
import Link from '../Link';
import StationThreadLink from '../StationThreadLink/StationThreadLink';

import stationKeyset from '../../i18n/station';

const b = B('StationScheduleBlock');

interface IStationScheduleBlockEl {
    scheduleBlock: IStationScheduleBlock;
    isAllDays: boolean;
    tld: Tld;
    language: Lang;
    stationFromId: number;
    whenDate: DateRobot | null;
    stopFilter: IStationStopFilter | null;

    className?: string;
}

export default memo(StationScheduleBlock);

function StationScheduleBlock({
    scheduleBlock,
    isAllDays,
    tld,
    language,
    stationFromId,
    whenDate,
    stopFilter,

    className,
}: IStationScheduleBlockEl): ReactElement | null {
    const {
        title,
        number,
        transportType,
        comment,
        schedule: schedules,
    } = scheduleBlock;

    const threadBlocks = schedules
        .map((schedule, index) => {
            // index должен оставаться постоянным для блока с нитками, иначе будут проблемы с перфомансом при фильтрации по станции
            const {
                threads: allThreads,
                daysText,
                runDaysText,
                exceptDaysText,
            } = schedule;

            const {threads, intervalThreads} = getScheduleThreads(
                allThreads,
                stopFilter,
            );

            if (threads.length === 0 && intervalThreads.length === 0) {
                return null;
            }

            return (
                <Fragment key={index}>
                    {isAllDays && (
                        <div className={b('runDays')}>
                            <StationRunDays
                                daysText={daysText ?? null}
                                runDaysText={runDaysText ?? null}
                                exceptDaysText={exceptDaysText ?? null}
                            />
                        </div>
                    )}

                    {/* Интервальные нитки */}
                    {intervalThreads.map(intervalThread => {
                        const {
                            interval: {beginTime, endTime, density},
                            canonicalUid,
                        } = intervalThread;

                        const url = getThreadUrl({
                            canonicalUid,
                            tld,
                            language,
                            stationFromId,
                            departureFromDateRobot: whenDate ?? undefined,
                        });

                        return (
                            <div
                                className={b('intervalSchedule')}
                                key={canonicalUid}
                            >
                                <Link
                                    href={url}
                                    className={b('intervalScheduleLink')}
                                >
                                    {stationKeyset('interval-thread-time', {
                                        beginTime,
                                        endTime,
                                    })}
                                </Link>
                                {` ${density}`}

                                {comment && (
                                    <span className={b('comment')}>
                                        {' '}
                                        {comment.trim().replace(' ', CHAR_NBSP)}
                                    </span>
                                )}
                            </div>
                        );
                    })}

                    {/* Таблица с нитками */}
                    {threads.length > 0 && (
                        <div className={b('tableScheduleContainer')}>
                            <div className={b('tableSchedule')}>
                                {threads.map(thread => (
                                    <StationThreadLink
                                        className={b('threadLink')}
                                        key={thread.canonicalUid}
                                        thread={thread}
                                        tld={tld}
                                        language={language}
                                        stationFromId={stationFromId}
                                    />
                                ))}
                            </div>
                        </div>
                    )}
                </Fragment>
            );
        })
        .filter(Boolean);

    if (threadBlocks.length === 0) {
        return null;
    }

    return (
        <div className={b({isAllDays}, className)}>
            <div className={b('title')}>
                <TransportIcon
                    className={b('transportIcon')}
                    transportType={transportType}
                />

                {number.length > 0 && (
                    <span className={b('number')}>№ {number}</span>
                )}

                <span className={b('name')}>{title}</span>
            </div>

            {threadBlocks}
        </div>
    );
}
