import React, {memo, ReactElement, useEffect} from 'react';
import B from 'bem-cn-lite';

import DateSpecialValue from '../../interfaces/date/DateSpecialValue';
import {ILink} from '../Link';
import IconGlyph from '../../interfaces/components/IconGlyph';

import {FilterTransportType, TransportType} from '../../lib/transportType';

import getSearchUrl from '../../lib/url/getSearchUrl';
import getValueFromEnum from '../../lib/enum/getValueFromEnum';
import useSelector from '../useSelector';
import {reachGoal, reachGoalOnce} from '../../lib/yaMetrika';

// eslint-disable-next-line no-duplicate-imports
import Link from '../Link';
import Icon from '../Icon/Icon';
import TransportIcon from '../TransportIcon/TransportIcon';

import stationKeyset from '../../i18n/station';

const b = B('StationWayToAirport');

function onAeroexpressLinkClick(): void {
    reachGoal('station_way_to_airport_click');
}

interface IStationWay {
    className?: string;
    linkColor?: ILink['colors'];
    shortText?: boolean;
    showIcon?: boolean;
}

export default memo(StationWayToAirport);

function StationWayToAirport({
    className,
    linkColor,
    shortText = false,
    showIcon = false,
}: IStationWay): ReactElement | null {
    const wayToAirport = useSelector(state => state.station.wayToAirport);
    const tld = useSelector(state => state.tld);
    const language = useSelector(state => state.language);
    const whenSpecial = useSelector(state => state.station.whenSpecial);
    const whenDate = useSelector(state => state.station.whenDate);

    useEffect(() => {
        if (wayToAirport) {
            reachGoalOnce('station_way_to_airport_shown');
        }
    }, [wayToAirport]);

    if (!wayToAirport) {
        return null;
    }

    return (
        <Link
            className={b(undefined, className)}
            href={getSearchUrl({
                fromPoint: wayToAirport.fromPoint,
                toPoint: wayToAirport.toPoint,
                fromSlug: wayToAirport.fromSlug,
                toSlug: wayToAirport.toSlug,
                tld,
                language,
                when: whenSpecial
                    ? getValueFromEnum(whenSpecial, DateSpecialValue)
                    : whenDate,
                transportType: FilterTransportType.suburban,
            })}
            onClick={onAeroexpressLinkClick}
            colors={linkColor}
        >
            {wayToAirport.wayType !== 'aeroexpress' && showIcon && (
                <TransportIcon
                    className={b('suburbanIcon')}
                    size="medium"
                    transportType={TransportType.suburban}
                />
            )}
            {wayToAirport.wayType === 'aeroexpress' && (
                <Icon
                    className={b('aeroexpressIcon')}
                    glyph={IconGlyph.aeroexpressLogo}
                />
            )}
            <span>
                {shortText && wayToAirport.wayType === 'aeroexpress'
                    ? stationKeyset('aeroexpress')
                    : wayToAirport.title}
            </span>
        </Link>
    );
}
