import React, {memo} from 'react';
import B from 'bem-cn-lite';

import {sort} from '../../lib/sort/sortStations';

import {TransportType} from '../../lib/transportType';
import IStationGroup from '../../interfaces/state/stationsGroup/IStationGroup';

import needShowList from './needShowList';
import getGroupLimit from './getGroupLimit';
import getStatusText from './getStatusText';
import getTitle from './getTitle';

import StationsGroupTitle from '../HomePage/StationsGroupTitle/StationsGroupTitle';
import Station from '../HomePage/Station/Station';
import InformationIcon from '../basic/InformationIcon';

import aeroexpressKeyset from '../../i18n/bt-lib-aeroexpress';

const b = B('StationsGroup');

interface IStationsGroupParams {
    group: IStationGroup;
}

export default memo(StationsGroup);

function StationsGroup({group}: IStationsGroupParams): React.ReactElement {
    const {stations, t_type: type} = group;

    const showList = needShowList(group);
    const hasPopup = stations.length > getGroupLimit(type);

    const sortedStations = sort(stations);

    const statusTextForFirst = getStatusText(stations[0]);
    const existsStatusTextForAnyStations = stations.some(station =>
        getStatusText(station),
    );
    const showAirportDisclaimer = Boolean(
        type === TransportType.plane &&
            stations.length &&
            existsStatusTextForAnyStations,
    );

    // Блок считается одиночным, если его станции спрятаны в попап или других станций/авиаэкспресса/дисклеймера у него нет
    const standalone =
        hasPopup || (!showAirportDisclaimer && stations.length === 1);

    return (
        <li className={b({type, standalone})}>
            <div className={b('content')}>
                <StationsGroupTitle
                    hasPopup={hasPopup}
                    title={getTitle(group)}
                    status={statusTextForFirst}
                    stations={sortedStations}
                    type={type}
                />

                {showList && (
                    <ul className={b('list')}>
                        {sortedStations.map(station => (
                            <Station
                                station={station}
                                key={station.id}
                                status={getStatusText(station)}
                            />
                        ))}
                    </ul>
                )}

                {showAirportDisclaimer && (
                    <div className={b('disclaimer-container')}>
                        <span className={b('disclaimer-text')}>
                            {aeroexpressKeyset('disclaimer')}
                        </span>
                        <InformationIcon
                            hint={aeroexpressKeyset('disclaimer-description')}
                            thin
                        />
                    </div>
                )}
            </div>
        </li>
    );
}
