import React, {memo, useCallback} from 'react';
import B from 'bem-cn-lite';
import {publish} from 'pubsub-js';

import {SEGMENTS} from '../../lib/constants/pubSubEvents';

import IStationGroup from '../../interfaces/state/stationsGroup/IStationGroup';
import {TransportType} from '../../lib/transportType';
import IconGlyph from '../../interfaces/components/IconGlyph';

import {sort} from '../../lib/sort/sortStations';
import {shouldShowAeroexpress} from '../../lib/aeroexpress';
import needShowList from './needShowList';
import getGroupLimit from './getGroupLimit';
import getStatusText from './getStatusText';
import getTitle from './getTitle';
import useDispatch from '../useDispatch';
import useSelector from '../useSelector';
import {useTld} from '../../hooks/useTld';
import {useLang} from '../../hooks/useLang';

import {
    setIsOpenedBusCollapse,
    setIsOpenedTrainCollapse,
} from '../../actions/home';

import StationsGroupTitle from '../HomePage/StationsGroupTitle/StationsGroupTitle';
import Station from '../HomePage/Station/Station';
import Aeroexpress from '../HomePage/Aeroexpress/Aeroexpress';
import InformationIcon from '../basic/InformationIcon';
import DirectionsTitle from '../HomePage/DirectionsTitle/DirectionsTitle';
import Collapse from '../Collapse';
import Icon from '../Icon/Icon';

import aeroexpressKeyset from '../../i18n/bt-lib-aeroexpress';

const b = B('StationsGroup');

interface IStationsGroupParams {
    group: IStationGroup;

    cityId?: number;
}

export default memo(StationsGroup);

function StationsGroup({
    group,

    cityId,
}: IStationsGroupParams): React.ReactElement {
    const {stations, t_type: type} = group;

    const showList = needShowList(group);

    const sortedStations = sort(stations);
    const showCollapse =
        (type === TransportType.bus || type === TransportType.train) &&
        stations.length > getGroupLimit(type);

    const onAirportDisclaimerClick = useCallback(() => {
        publish(SEGMENTS.HOMEPAGE_AERO_INFO, {});
    }, []);

    const isOpenedBusCollapse = useSelector(
        state => state.home.isOpenedBusCollapse,
    );
    const isOpenedTrainCollapse = useSelector(
        state => state.home.isOpenedTrainCollapse,
    );

    const tld = useTld();
    const language = useLang();
    const dispatch = useDispatch();

    const onClick = useCallback(() => {
        if (type === TransportType.bus) {
            dispatch(setIsOpenedBusCollapse(!isOpenedBusCollapse));
        }

        if (type === TransportType.train) {
            dispatch(setIsOpenedTrainCollapse(!isOpenedTrainCollapse));
        }
    }, [isOpenedBusCollapse, isOpenedTrainCollapse, type, dispatch]);

    if (showCollapse) {
        const isOpen =
            type === TransportType.bus
                ? isOpenedBusCollapse
                : type === TransportType.train
                ? isOpenedTrainCollapse
                : false;

        return (
            <li className={b({type, isOpen, collapsed: true})} key={type}>
                <Collapse
                    opened={isOpen}
                    buttonElement={
                        <div className={b('titleContainer')}>
                            <DirectionsTitle
                                icon={IconGlyph[type]}
                                className={b('title', {isMobile: true})}
                            >
                                {getTitle(group)}
                            </DirectionsTitle>
                            <Icon
                                className={b('arrow')}
                                glyph={IconGlyph.arrowDown}
                            />
                        </div>
                    }
                    contentElement={
                        <ul className={b('stations')}>
                            {sortedStations.map(station => (
                                <Station
                                    station={station}
                                    key={station.id}
                                    status={getStatusText(station)}
                                    usePopularTitle
                                />
                            ))}
                        </ul>
                    }
                    onClick={onClick}
                />
            </li>
        );
    }

    const statusTextForFirst = getStatusText(stations[0]);
    const existsStatusTextForAnyStations = stations.some(station =>
        getStatusText(station),
    );
    const showAirportDisclaimer = Boolean(
        type === TransportType.plane &&
            stations.length &&
            existsStatusTextForAnyStations,
    );
    const showAeroexpress = shouldShowAeroexpress(stations, type);
    // Блок считается одиночным, если его станции спрятаны в попап или других станций/авиаэкспресса/дисклеймера у него нет
    const standalone =
        !showAeroexpress && !showAirportDisclaimer && stations.length === 1;

    return (
        <li className={b({type, standalone})} key={type}>
            <div className={b('content')}>
                <StationsGroupTitle
                    title={getTitle(group)}
                    status={statusTextForFirst}
                    stations={sortedStations}
                    type={type}
                />

                {showList && (
                    <ul className={b('list')}>
                        {sortedStations.map(station => (
                            <Station
                                station={station}
                                key={station.id}
                                status={getStatusText(station)}
                            />
                        ))}
                    </ul>
                )}

                {showAeroexpress && (
                    <Aeroexpress
                        stations={stations}
                        cityId={cityId}
                        language={language}
                        tld={tld}
                    />
                )}

                {showAirportDisclaimer && (
                    <div
                        className={b('disclaimer-container')}
                        onClick={onAirportDisclaimerClick}
                    >
                        <span className={b('disclaimer-text')}>
                            {aeroexpressKeyset('disclaimer')}
                        </span>
                        <InformationIcon
                            hint={aeroexpressKeyset('disclaimer-description')}
                            thin
                            inactive
                        />
                    </div>
                )}
            </div>
        </li>
    );
}
