import {React, PureComponent, B, mix, PropTypes, refCallback} from '../base';

import noop from '../../lib/noop';

const b = B('Sticky');

const initialState = {
    fixed: false,
    anchorHeight: 0,
};

/*
 * Компонент-обертка, который "прилепляется" к верху экрана, как только область просмотра
 * браузера будет ниже данного компонента
 */
export default class Sticky extends PureComponent {
    checkInterval = null; // Дополнительный интервал по времени для проверки позиции элемента

    static propTypes = {
        className: PropTypes.string,
        static: PropTypes.bool,
        contentClassName: PropTypes.string,
        contentStyle: PropTypes.object,
        contentRef: PropTypes.func,
        onChange: PropTypes.func,
    };

    static defaultProps = {
        className: '',
        static: false,
        contentClassName: '',
        contentStyle: {},
        contentRef: noop,
        onChange: noop,
    };

    constructor(props) {
        super(props);
        this.state = initialState;
    }

    componentDidMount() {
        window.addEventListener('scroll', this.onScroll);

        this.checkInterval = setInterval(this.onScroll, 1000);

        this.onScroll();
    }

    componentWillReceiveProps(nextProps) {
        if (nextProps.static && nextProps.static !== this.props.static) {
            this.setState(initialState);
        }
    }

    componentWillUnmount() {
        window.removeEventListener('scroll', this.onScroll);

        clearInterval(this.checkInterval);
    }

    onScroll = () => {
        if (this.props.static) {
            return null;
        }

        requestAnimationFrame(this.calculatePosition);
    };

    calculatePosition = () => {
        const anchorTop = this._anchor.getBoundingClientRect().top;

        if (anchorTop <= 0) {
            if (!this.state.fixed) {
                this.setState({
                    fixed: true,
                    anchorHeight: this._content.offsetHeight,
                });
                this.props.onChange(true);
            }
        } else if (this.state.fixed) {
            this.setState({
                fixed: false,
                anchorHeight: 0,
            });
            this.props.onChange(false);
        }
    };

    refCallbackForContent = ref => {
        refCallback(this, '_content')(ref);
        this.props.contentRef(ref);
    };

    render() {
        const {className, contentClassName, contentStyle, children} =
            this.props;

        return (
            <div className={mix(b({fixed: this.state.fixed}), className)}>
                <div
                    className={b('anchor')}
                    ref={refCallback(this, '_anchor')}
                    style={{height: this.state.anchorHeight}}
                />
                <div
                    className={mix(b('content'), contentClassName)}
                    ref={this.refCallbackForContent}
                    style={contentStyle}
                >
                    {children}
                </div>
            </div>
        );
    }
}
