import React, {useContext} from 'react';
import B from 'bem-cn-lite';

import ISubSegment from '../../interfaces/segment/ISubSegment';
import ISearchContext from '../../interfaces/state/search/ISearchContext';

import {isCity} from '../../lib/point/pointType';
import getThreadUrlForSegment from '../../lib/segments/getThreadUrlForSegment';
import {
    getLocalDepartureTime,
    getLocalArrivalTime,
} from '../../lib/segments/cachedSegmentTime';
import useSelector from '../useSelector';
import {useFlags} from '../../hooks/useFlags';

import SegmentTitleLinkUtmContext from '../../contexts/SegmentTitleLinkUtmContext';
import SegmentTitle from '../SegmentTitle/SegmentTitle.mobile';
import SegmentTimeAndStations from '../SegmentTimeAndStations/SegmentTimeAndStations.mobile';

const b = B('SubSegment');

interface ISubSegmentParameters {
    segment: ISubSegment;
    context: ISearchContext;
    isSuburbanSearchResult: boolean;
    isAllDaysSearch: boolean;

    showTransportIcon?: boolean;
    showArrivalDate?: boolean;
    showDepartureDate?: boolean;
    className?: string;
}

export default React.memo(SubSegment);

function SubSegment({
    segment,
    context,
    isSuburbanSearchResult,
    isAllDaysSearch,

    showTransportIcon = false,
    showArrivalDate = false,
    showDepartureDate = false,
    className,
}: ISubSegmentParameters): React.ReactElement {
    const seoQueryParams = useSelector(state => state.seoQueryParams);
    const tld = useSelector(state => state.tld);
    const language = useSelector(state => state.language);
    const clientId = useSelector(state => state.user.clientId);
    const flags = useFlags();
    const isProduction = useSelector(state => state.environment.production);

    const {getUtmMedium} = useContext(SegmentTitleLinkUtmContext);

    const {
        transport,
        title,
        number,
        thread,
        isThroughTrain,
        isInterval,
        departureMoment = getLocalDepartureTime(segment),
        arrivalMoment = getLocalArrivalTime(segment),
        stationFrom,
        stationTo,
        duration,
        isFuzzyFrom,
        isFuzzyTo,
        company,
    } = segment;

    const {code: transportType} = transport;

    const threadUrl = getThreadUrlForSegment({
        segment,
        tld,
        language,
        seoQueryParams,
        clientId,
        utmMedium: getUtmMedium(transportType),
        flags,
        isProduction,
        isToCitySearchContext: isCity(context.to),
    });
    const isExpress = thread ? thread.isExpress : false;

    return (
        <div className={b({}, className)}>
            <SegmentTitle
                transport={transport}
                title={title}
                isSuburbanSearchResult={isSuburbanSearchResult}
                number={number}
                showTransportIcon={showTransportIcon}
                thread={thread || undefined}
                isThroughTrain={isThroughTrain}
                company={company}
                threadUrl={threadUrl}
                linkTitleColorsInherit
                showTitle
            />

            <SegmentTimeAndStations
                className={b('timeAndStations')}
                transportType={transportType}
                isExpress={isExpress}
                isInterval={isInterval}
                departureMoment={departureMoment}
                arrivalMoment={arrivalMoment}
                stationFrom={stationFrom}
                stationTo={stationTo}
                showArrivalDate={showArrivalDate}
                showDepartureDate={showDepartureDate}
                isAllDaysSearch={isAllDaysSearch}
                duration={duration}
                isFuzzyFrom={isFuzzyFrom}
                isFuzzyTo={isFuzzyTo}
            />
        </div>
    );
}
