import React, {PureComponent} from 'react';
import B from 'bem-cn-lite';
import throttle from 'lodash/throttle';

import {refCallback} from '../base';

import IconGlyph from '../../interfaces/components/IconGlyph';

import defaultDataProvider from '../../lib/suggests/defaultDataProvider';
import {reachGoal} from '../../lib/yaMetrika';
import {ceilByStep} from '../../lib/mathUtils';
import noop from '../../lib/noop';

import Icon from '../Icon/Icon';
import ModalHistoryWrapper from '../ModalHistoryWrapper/ModalHistoryWrapper';

import keyset from '../../i18n/search-form';

import ClearButton from '../basic/ClearButton';

const b = B('Suggest');

class SuggestItem extends PureComponent {
    constructor(props) {
        super(props);
    }

    onClick = () => {
        this.props.onClick(this.props.value);
    };

    render() {
        const {title, subtitle} = this.props;

        return (
            <div className={b('list-item')} onClick={this.onClick}>
                <span className={b('list-item-title')}>{title}</span>
                <span className={b('list-item-subtitle')}>{subtitle}</span>
            </div>
        );
    }
}

export default class Suggest extends PureComponent {
    constructor(props) {
        super(props);

        this.state = {
            list: [],
            inited: false,

            // Данные для метрики
            fetchedAt: null, // время получения данных из саджестов
        };

        this.suggest = throttle(this.suggest.bind(this), 200);
    }

    componentDidMount() {
        const {props} = this;
        const {inputData, suggestField} = props;

        this.suggest(inputData.userInput[suggestField].title);
        setTimeout(() => {
            this.setState({inited: true});
        }, 0);
        setTimeout(() => {
            this.focusOnField();
        }, 1);
    }

    componentWillReceiveProps(newProps) {
        this.suggest(newProps.inputData.userInput[newProps.suggestField].title);
    }

    onClose = () => this.props.hide();

    onSelect = value => {
        const {fetchedAt} = this.state;
        const {
            select,
            inputData: {userInput},
            suggestField,
        } = this.props;

        reachGoal('search_form_change_parameters');
        reachGoal('point_suggest_selected', {
            timeToSelectFromSuggests: fetchedAt
                ? ceilByStep(Date.now() - fetchedAt, 200)
                : null,
            selectedObjectKey: value.key,
            searchFormCurrentValue: userInput[suggestField].title,
        });

        this.setState({
            fetchedAt: null,
        });

        select(value);
    };

    onClearClick = () => {
        const {onChange, suggestField} = this.props;

        onChange(suggestField, {
            title: '',
            key: '',
        });

        this.focusOnField();
    };

    onChange = e => {
        const {suggestField, onChange} = this.props;

        const data = {
            title: e.target.value,
            key: '',
        };

        onChange(suggestField, data);
    };

    onSubmit = e => {
        const {onSubmit} = this.props;

        e.preventDefault();
        onSubmit();
    };

    focusOnField = () => {
        const {suggestField} = this.props;
        const input = this[`_${suggestField}`];

        if (input) {
            const lengthValue = input.value.length;

            input.focus();
            input.setSelectionRange(lengthValue, lengthValue);
        }
    };

    suggest(text) {
        const {dataProvider, setupSuggests, dataProviderOptions} = this.props;

        dataProvider(text, dataProviderOptions)
            .then(list => (setupSuggests ? setupSuggests(list) : list))
            .then(list => {
                this.setState(() => ({
                    list,
                    fetchedAt: Date.now(),
                }));
            })
            .catch(noop);
    }

    renderInput() {
        const {suggestField, inputData, formKeyset} = this.props;
        const {userInput} = inputData;

        const placeholder = formKeyset(`label-${suggestField}`);

        return (
            <div className={b('inputContainer')}>
                <form onSubmit={this.onSubmit}>
                    <input
                        className={b('input')}
                        ref={refCallback(this, `_${suggestField}`)}
                        name={`${suggestField}Name`}
                        value={userInput[suggestField].title}
                        onChange={this.onChange}
                        placeholder={placeholder}
                        autoComplete="off"
                    />
                </form>
                <ClearButton
                    className={b('clearButton')}
                    onClick={this.onClearClick}
                    visible={Boolean(
                        suggestField &&
                            userInput[suggestField].title.length > 0,
                    )}
                />
            </div>
        );
    }

    render() {
        const {state, props} = this;

        const {hide, suggestField, fixedPosition, formKeyset} = props;

        const {list, inited} = state;

        return (
            <ModalHistoryWrapper onClose={this.onClose}>
                <div className={b({fixedPosition, inited})}>
                    <div className={b('title')}>
                        {formKeyset(`label-${suggestField}`)}
                    </div>
                    <Icon
                        glyph={IconGlyph.cross}
                        className={b('close')}
                        onClick={hide}
                    />
                    {this.renderInput()}
                    <div className={b('list', {visible: Boolean(list.length)})}>
                        {list.map(i => (
                            <SuggestItem
                                key={i.value.key}
                                title={i.value.title}
                                subtitle={i.text}
                                value={i.value}
                                onClick={this.onSelect}
                            />
                        ))}
                    </div>
                </div>
            </ModalHistoryWrapper>
        );
    }
}

Suggest.defaultProps = {
    formKeyset: keyset,
    dataProvider: defaultDataProvider,
    dataProviderOptions: {
        url: '//suggests.rasp.yandex.net',
        path: 'all_suggests',
        query: {
            format: 'old',
            lang: 'ru',
            client_city: 213,
            national_version: 'ru',
        },
    },
    onSubmit: noop,
};
