import React, {
    ReactElement,
    useState,
    useCallback,
    useMemo,
    useEffect,
    useRef,
} from 'react';
import B from 'bem-cn-lite';

import ISuggestData from '../../interfaces/components/suggest/ISuggestData';
import ISuggestElement from '../../interfaces/components/suggest/ISuggestElement';
import PrimaryPosition from '../../interfaces/lib/dimensions/PrimaryPosition';
import SecondaryPosition from '../../interfaces/lib/dimensions/SecondaryPosition';
import KeyCode from '../../interfaces/KeyCode';

import typedMemo from '../typedMemo';
import useGetData from '../useGetData';

import Popup from '../Popup/Popup';
import Input from '../Input/Input';
import WrapSubstring from '../Wrap/WrapSubstring';

const b = B('SuggestDesktop');

const suggestPositions = [
    [PrimaryPosition.bottom, SecondaryPosition.left],
    [PrimaryPosition.top, SecondaryPosition.left],
];

interface ISuggestDesktop<SuggestElement extends ISuggestElement> {
    value: string;
    onChange: (value: string) => void;
    getData: (
        value: string,
    ) => ISuggestData<SuggestElement> | Promise<ISuggestData<SuggestElement>>;
    onClickElement: (element: SuggestElement) => void;

    onReset?: () => void; // Обработчик на клик по крестику в инпуте
    placeholder?: string;
    limit?: number; // Максимальное количество отображаемых саджестов
    regExpForHighlightingElements?: RegExp; // Регулярка для выделения совпавшего текста в саджестах
    className?: string;
}

export default typedMemo(SuggestDesktop);

function SuggestDesktop<SuggestElement extends ISuggestElement>({
    value,
    onChange,
    getData,
    onClickElement,

    onReset,
    placeholder,
    limit = 5,
    regExpForHighlightingElements,
    className,
}: ISuggestDesktop<SuggestElement>): ReactElement {
    const [focused, setFocused] = useState(false);

    const inputRef = useRef<HTMLInputElement | null>(null);

    const suggestData = useGetData(value, getData);
    const elements = suggestData?.elements || [];

    // Хак для того, чтобы не было артефактов при исчезании попапа, когда изменилась строка поиска
    // и сначала в попапе исчезают саджесты, а потом исчезаеи сам попап
    const memoState = useMemo<{lastFilledElements: SuggestElement[]}>(
        () => ({lastFilledElements: []}),
        [],
    );

    // Запоминаем последние полученные саджесты, когда на поле был фокус
    memoState.lastFilledElements =
        elements.length && focused ? elements : memoState.lastFilledElements;

    // Если инпут в фокусе, то показываем актуальные саджесты, если нет, то
    // берем предыдущие, чтобы пока саджесты пропадают (fadeOut блока Popup), они не менялись
    const elementsForDisplay =
        elements.length && focused ? elements : memoState.lastFilledElements;

    const suggestIsVisible = focused && elements.length > 0;

    // Управление hover-ом
    const [hoverState, setHoverState] = useState<{
        forElements?: SuggestElement[];
        element?: SuggestElement;
    }>({
        forElements: suggestData?.elements,
    });

    const setHoverOnElement = useCallback(
        (element?: SuggestElement): void => {
            setHoverState({
                forElements: elements,
                element,
            });
        },
        [elements],
    );

    const onFocusInput = useCallback(() => {
        setFocused(true);
        // Сбрасываем позицию ховера
        setHoverOnElement();
    }, [setHoverOnElement]);

    const onBlurInput = useCallback(() => {
        setFocused(false);
    }, []);

    const onKeyPress = useCallback(
        (e: KeyboardEvent): void => {
            const pressedKey = e.which || e.keyCode;

            const currentHoverElement =
                hoverState.forElements === elements
                    ? hoverState.element
                    : undefined;

            switch (pressedKey) {
                case KeyCode.enter:
                    e.preventDefault();

                    if (suggestIsVisible && currentHoverElement) {
                        // Вызываем полтерю фокуса на инпуте, так как сделан выбор из саджестов
                        // eslint-disable-next-line no-unused-expressions
                        inputRef.current?.blur();

                        onClickElement(currentHoverElement);
                    }

                    break;
                case KeyCode.up: {
                    e.preventDefault();

                    if (!suggestIsVisible) {
                        return;
                    }

                    const hoverIndex = elements.findIndex(
                        el => el === currentHoverElement,
                    );
                    const newHoverIndex =
                        hoverIndex > 0 ? hoverIndex - 1 : elements.length - 1;

                    setHoverOnElement(elements[newHoverIndex]);

                    break;
                }

                case KeyCode.down: {
                    e.preventDefault();

                    if (!suggestIsVisible) {
                        return;
                    }

                    const hoverIndex = elements.findIndex(
                        el => el === currentHoverElement,
                    );
                    const newHoverIndex =
                        hoverIndex >= 0 && hoverIndex < elements.length - 1
                            ? hoverIndex + 1
                            : 0;

                    setHoverOnElement(elements[newHoverIndex]);

                    break;
                }

                default: {
                    return;
                }
            }
        },
        [
            elements,
            hoverState.element,
            hoverState.forElements,
            onClickElement,
            setHoverOnElement,
            suggestIsVisible,
        ],
    );

    useEffect(() => {
        if (focused) {
            document.addEventListener('keydown', onKeyPress);
        }

        return () => {
            document.removeEventListener('keydown', onKeyPress);
        };
    }, [focused, onKeyPress]);

    return (
        <span className={b(undefined, className)}>
            <Input
                className={b('input')}
                value={value}
                onChange={onChange}
                placeholder={placeholder}
                withSearchIcon
                withReset
                widthMod="100"
                onFocus={onFocusInput}
                onBlur={onBlurInput}
                onReset={onReset}
                inputRef={inputRef}
            />

            <Popup
                className={b('suggests')}
                visible={suggestIsVisible}
                positions={suggestPositions}
                withoutArrow
                theme="travel"
                contentWidth="full"
            >
                {elementsForDisplay.slice(0, limit).map(element => {
                    const {title, subtitle} = element;

                    return (
                        <div
                            className={b('element', {
                                withoutSubtitle: !subtitle,
                                hovered:
                                    elements === hoverState.forElements &&
                                    element === hoverState.element,
                            })}
                            key={`${title}--${subtitle}`}
                            // eslint-disable-next-line react/jsx-no-bind
                            onClick={() => onClickElement(element)}
                            // eslint-disable-next-line react/jsx-no-bind
                            onMouseOver={() => setHoverOnElement(element)}
                        >
                            <div className={b('elementTitle')}>
                                {regExpForHighlightingElements ? (
                                    <WrapSubstring
                                        str={title}
                                        regExp={regExpForHighlightingElements}
                                        className={b('bold')}
                                    />
                                ) : (
                                    title
                                )}
                            </div>

                            {subtitle && (
                                <div className={b('elementSubtitle')}>
                                    {regExpForHighlightingElements ? (
                                        <WrapSubstring
                                            str={subtitle}
                                            regExp={
                                                regExpForHighlightingElements
                                            }
                                            className={b('bold')}
                                        />
                                    ) : (
                                        subtitle
                                    )}
                                </div>
                            )}
                        </div>
                    );
                })}
            </Popup>
        </span>
    );
}
